/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Martin Polak
 *   Co-authors:
 *      ...
 *
 */

#include "WeaponPack.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"

#include "Weapon.h"
#include "WeaponSystem.h"
#include "DefaultWeaponmodeLink.h"

namespace orxonox
{
    RegisterClass(WeaponPack);

    WeaponPack::WeaponPack(Context* context) : BaseObject(context)
    {
        RegisterObject(WeaponPack);

        this->weaponSystem_ = nullptr;
    }

    WeaponPack::~WeaponPack()
    {
        if (this->isInitialized())
        {
            if( this->weaponSystem_ )
                this->weaponSystem_->removeWeaponPack(this);

            while (!this->weapons_.empty())
                (*this->weapons_.begin())->destroy();

            for (std::set<DefaultWeaponmodeLink*>::iterator it = this->links_.begin(); it != this->links_.end(); )
                (*(it++))->destroy();
        }
    }

    void WeaponPack::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(WeaponPack, XMLPort, xmlelement, mode);

        XMLPortObjectExtended(WeaponPack, Weapon, "", addWeapon, getWeapon, xmlelement, mode, false, false);
        XMLPortObject(WeaponPack, DefaultWeaponmodeLink, "links", addDefaultWeaponmodeLink, getDefaultWeaponmodeLink, xmlelement, mode);
    }

    /**
    @brief
        Fire all weapons in this WeaponSet with the defined weaponmode.
    */
    void WeaponPack::fire(unsigned int weaponmode)
    {
        for (Weapon* weapon : this->weapons_)
            weapon->fire(weaponmode);
    }

    /**
    @brief
        Reload all weapons in this WeaponSet.
    */
    void WeaponPack::reload()
    {
        for (Weapon* weapon : this->weapons_)
            weapon->reload();
    }

    void WeaponPack::addWeapon(Weapon * weapon)
    {
        if (!weapon)
            return;

        this->weapons_.push_back(weapon);
        weapon->setWeaponPack(this);
    }

    void WeaponPack::removeWeapon(Weapon * weapon)
    {
        if (!weapon)
            return;

        std::vector<Weapon*>::iterator it = std::find(this->weapons_.begin(), this->weapons_.end(), weapon);
        assert(it != this->weapons_.end());
        this->weapons_.erase(it);
        weapon->setWeaponPack(nullptr);
    }

    Weapon * WeaponPack::getWeapon(unsigned int index) const
    {
        unsigned int i = 0;

        for (Weapon* weapon : this->weapons_)
        {
            if (i == index)
                return weapon;
            ++i;
        }

        return nullptr;
    }

    void WeaponPack::addDefaultWeaponmodeLink(DefaultWeaponmodeLink* link)
    {
        this->links_.insert(link);
    }

    DefaultWeaponmodeLink* WeaponPack::getDefaultWeaponmodeLink(unsigned int index) const
    {
        unsigned int i = 0;
        for (DefaultWeaponmodeLink* link : this->links_)
        {
            if (i == index)
                return link;

            ++i;
        }
        return nullptr;
    }

    unsigned int WeaponPack::getDesiredWeaponmode(unsigned int firemode) const
    {
        for (DefaultWeaponmodeLink* link : this->links_)
            if (link->getFiremode() == firemode)
                return link->getWeaponmode();

        return WeaponSystem::WEAPON_MODE_UNASSIGNED;
    }

    void WeaponPack::notifyWeapons()
    {
        for (Weapon* weapon : this->weapons_)
            weapon->setWeaponPack(this);
    }

    void WeaponPack::updateMunition()
    {
        for (Weapon* weapon : this->weapons_)
            weapon->updateMunition();
    }
}
