/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Authors:
 *      Martin Polak
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      Fabien Vultier
 *
 */

#ifndef _Munition_H__
#define _Munition_H__

#include "OrxonoxPrereqs.h"

#include <map>
#include "core/BaseObject.h"
#include "tools/Timer.h"

namespace orxonox
{
    enum class MunitionDeployment
    {
        Separate, // Every comsuming weapon mode has its own magazine. It is possible that one weapon mode is out of ammo while another still has some.
        Share, // All comsuming weapon modes take their munition from the same magazine. If this magazine is empty a new one is loaded.
        Stack // There is only one magazine where all the munition is stored. Use this deployment mode for heavy weapons loke rockets, bombs, ...
    };

    class _OrxonoxExport Munition : public BaseObject
    {        
        struct Magazine
        {
            public:
                Magazine(Munition* munition, bool bUseReloadTime = true);
                virtual ~Magazine() {}

                unsigned int munition_;
                Timer loadTimer_;
                bool bLoaded_;

            private:
                void loaded(Munition* munition);
        };

        public:
            Munition(Context* context);
            virtual ~Munition();

            virtual void XMLPort(Element& xmlelement, XMLPort::Mode mode) override;

            unsigned int getNumMunition(WeaponMode* user) const;
            unsigned int getNumMunitionInCurrentMagazine(WeaponMode* user) const;
            unsigned int getNumMagazines() const;

            unsigned int getMaxMunition() const;
            inline unsigned int getMaxMagazines() const
                { return this->maxMagazines_; }
            inline unsigned int getMaxMunitionPerMagazine() const
                { return this->maxMunitionPerMagazine_; }
            inline MunitionDeployment getMunitionDeployment() const
                { return deployment_; }


            bool canTakeMunition(unsigned int amount, WeaponMode* user) const;
            bool takeMunition(unsigned int amount, WeaponMode* user);

            bool canReload() const;
            bool needReload(WeaponMode* user) const;
            bool reload(WeaponMode* user, bool bUseReloadTime = true);
            inline float getReloadTime() const
                { return this->reloadTime_; }

            bool canAddMunition(unsigned int amount) const;
            bool addMunition(unsigned int amount);

            bool canAddMagazines(unsigned int amount) const;
            /**
            @brief Try to add magazines.
            @param amount The amount of magazines tried to add.
            @return The amount of magazines sucessfully added.
            */
            unsigned int addMagazines(unsigned int amount);

            bool canRemoveMagazines(unsigned int amount) const;
            bool removeMagazines(unsigned int amount);

            bool dropMagazine(WeaponMode* user);

        protected:
            unsigned int maxMunitionPerMagazine_;
            unsigned int maxMagazines_;
            unsigned int unassignedMagazines_; // Number of magazines that are not assigned to a weapon mode. These are alway treated as full.
            std::map<WeaponMode*, Magazine*> assignedMagazines_; // Maps weapon modes to magazines that are currently used.

            MunitionDeployment deployment_; // Defines the behaviour how munition and magazines are distributed to the consuming weapon modes.

            bool bAllowMunitionRefilling_;
            bool bAllowMultiMunitionRemovementUnderflow_;

            float reloadTime_; // The time needed to replace a magazine by a new one.
            WeaponMode* lastFilledWeaponMode_; // Pointer to the weapon mode that got the last munition during the last call of addMunition.

        private:
            Magazine* getMagazine(WeaponMode* user) const;
            inline void setMaxMagazines(unsigned int maxMagazines)
                { this->maxMagazines_ = maxMagazines; }
            inline void setMaxMunitionPerMagazine(unsigned int maxMunitionPerMagazine)
                { this->maxMunitionPerMagazine_ = maxMunitionPerMagazine; }
            void setNumMagazines(unsigned int numMagazines);
    };
}

#endif /* _Munition_H__ */
