/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Oliver Scheuss
 *   Co-authors:
 *      simonmie
 *
 */

/**
    @file RocketOld.h
    @brief Definition of the RocketOld class.
*/

#ifndef _RocketOld_H__
#define _RocketOld_H__

#include "weapons/WeaponsPrereqs.h"

#include "tools/Timer.h"

#include "interfaces/RadarViewable.h"
#include "worldentities/ControllableEntity.h"

#include "BasicProjectile.h"

namespace orxonox
{
    class ConeCollisionShape;

    /**
    @brief
        RocketOld that can be steered by the player.

    @author
        Oli Scheuss
    @ingroup WeaponsProjectiles
    */
    class _WeaponsExport RocketOld : public ControllableEntity, public BasicProjectile, public RadarViewable
    {
        public:
            RocketOld(Context* context);
            virtual ~RocketOld();

            virtual void tick(float dt) override; //!< Defines which actions the RocketOld has to take in each tick.

            virtual bool collidesAgainst(WorldEntity* otherObject, const btCollisionShape* cs, btManifoldPoint& contactPoint) override;
            virtual void destroyObject(void) override;
            void destructionEffect();

            virtual void moveFrontBack(const Vector2& value) override {}
            virtual void moveRightLeft(const Vector2& value) override {}
            virtual void moveUpDown(const Vector2& value) override {}

            virtual void rotateYaw(const Vector2& value) override;
            virtual void rotatePitch(const Vector2& value) override;
            virtual void rotateRoll(const Vector2& value) override;

            /**
            @brief Moves the RocketOld in the Front/Back-direction by the specifed amount.
            @param value  The amount by which the RocketOld is to be moved.
            */
            inline void moveFrontBack(float value)
                { this->moveFrontBack(Vector2(value, 0)); }
            /**
            @brief Moves the RocketOld in the Right/Left-direction by the specifed amount.
            @param value  The amount by which the RocketOld is to be moved.
            */
            inline void moveRightLeft(float value)
                { this->moveRightLeft(Vector2(value, 0)); }
            /**
            @brief Moves the RocketOld in the Up/Down-direction by the specifed amount.
            @param value  The amount by which the RocketOld is to be moved.
            */
            inline void moveUpDown(float value)
                { this->moveUpDown(Vector2(value, 0)); }

            /**
            @brief Rotates the RocketOld around the y-axis by the specifed amount.
            @param value  The amount by which the RocketOld is to be rotated.
            */
            inline void rotateYaw(float value)
                { this->rotateYaw(Vector2(value, 0)); }
            /**
            @brief Rotates the RocketOld around the x-axis by the specifed amount.
            @param value  The amount by which the RocketOld is to be rotated.
            */
            inline void rotatePitch(float value)
                { this->rotatePitch(Vector2(value, 0)); }
            /**
            @brief Rotates the RocketOld around the z-axis by the specifed amount.
            @param value  The amount by which the RocketOld is to be rotated.
            */
            inline void rotateRoll(float value)
                { this->rotateRoll(Vector2(value, 0)); }

            virtual void setShooter(Pawn* shooter) override;

            virtual void fired(unsigned int firemode) override;

        private:
            Vector3 localAngularVelocity_; //!< Variable to temporarily store accumulated steering command input.

            WeakPtr<PlayerInfo> player_; //!< The player that controls the RocketOld.
            //WeakPtr<Pawn> pawn_; //!< The pawn that controls the RocketOld. TODO
            Timer destroyTimer_; //!< Timer to destroy the projectile after its lifetime has run out.
            float lifetime_; //!< The time the projectile exists.

            WorldSound* defSndWpnEngine_; //!< Engine sound.
            WorldSound* defSndWpnLaunch_; //!< Launch sound.
    };

}

#endif /* _RocketOld_H__ */
