/*
*   ORXONOX - the hottest 3D action shooter ever to exist
*                    > www.orxonox.net <
*
*
*   License notice:
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation; either version 2
*   of the License, or (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*   Author:
*     Fabien Vultier
*   Co-authors:
*      ...
*
*/

/**
    @file IceGunFreezer.h
    @brief Implementation of the IceGunFreezer class.
*/

#include "core/CoreIncludes.h"
#include "core/command/Executor.h"

#include "IceGunFreezer.h"

namespace orxonox
{
    RegisterClass(IceGunFreezer);

    /**
    @brief
        Constructor.
    */
    IceGunFreezer::IceGunFreezer(Context* context) : StaticEntity(context)
    {
        RegisterObject(IceGunFreezer);

        model = new Model(this->getContext());
        model->setMeshSource("IceStar.mesh");
        model->setScale(4.0);
        this->attach(model);
        model->setPosition(Vector3(0,0,0));
    }

    IceGunFreezer::~IceGunFreezer()
    {

    }

    /**
    @brief
        Sets the freeze time variable to the passed value.
    */
    void IceGunFreezer::setFreezeTime(float freezeTime)
    {
        freezeTime_ = freezeTime;
    }

    /**
    @brief
        Sets the freeze factor variable to the passed value.
    */
    void IceGunFreezer::setFreezeFactor(float freezeFactor)
    {
        freezeFactor_ = freezeFactor;
    }    

    /**
    @brief
        Try to slow down the WorldEntity where this is attached to. It is only possible to slow down a SpaceShip.
    */
    void IceGunFreezer::startFreezing()
    {
        WorldEntity* parent = this->getParent();

        // Check if the freezer is attached to a parent and check if the parent is a SpaceShip
        if (parent != nullptr && parent->isA(Class(SpaceShip)))
        {
            freezedSpaceShip_ = orxonox_cast<SpaceShip*>(parent);
            //Slow down the SpaceShip
            freezedSpaceShip_->addSpeedFactor(freezeFactor_);
        }

        // Start timer even if the victim is not a SpaceShip to avoid that the IceGunFreezer gets never destroyed if it collided against a Pawn
        this->freezeTimer_.setTimer(this->freezeTime_, false, createExecutor(createFunctor(&IceGunFreezer::stopFreezing, this)));
    }

    /**
    @brief
        This method is called by the timer. It gives back the original engine strength to the hit SpaceShip.
    */
    void IceGunFreezer::stopFreezing()
    {
        if (freezedSpaceShip_ != nullptr && freezeFactor_ != 0.0)
        {
            freezedSpaceShip_->addSpeedFactor(1/freezeFactor_);
        }

        this->destroy();
    }    
}
