/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file BoostPickup.cc
    @brief Implementation of the BoostPickup class.
*/

#include "BoostPickup.h"

#include <sstream>
#include "core/CoreIncludes.h"
#include "core/XMLPort.h"

#include "worldentities/pawns/SpaceShip.h"

namespace orxonox
{
    RegisterClass(BoostPickup);

    /**
    @brief
        Constructor. Registers the object and initializes the member variables.
    */
    BoostPickup::BoostPickup(Context* context) : Pickup(context)
    {
        RegisterObject(BoostPickup);

        this->initialize();
    }

    /**
    @brief
        Destructor.
    */
    BoostPickup::~BoostPickup()
    {
    }

    /**
    @brief
        Initializes the member variables.
    */
    void BoostPickup::initialize(void)
    {
        this->boostRefill_ = 0.0f;
        //Defines who is allowed to pick up the pickup.
        this->addTarget(ClassIdentifier<SpaceShip>::getIdentifier());
    }

    /**
    @brief
        Method for creating a BoostPickup object through XML.
    */
    void BoostPickup::XMLPort(Element& xmlelement, orxonox::XMLPort::Mode mode)
    {
        SUPER(BoostPickup, XMLPort, xmlelement, mode);

        XMLPortParam(BoostPickup, "boostrefill", setBoostRefill, getBoostRefill, xmlelement, mode);
    }

    /**
    @brief
        Set the boost refill of this pickup.
    */
    void BoostPickup::setBoostRefill(float boostRefill)
    {
        if(boostRefill >= 0.0f && boostRefill <= 1.0f)
        {
            this->boostRefill_ = boostRefill;
        }
    }

    /**
    @brief
        Is called when the pickup has transisted from used to unused or the other way around.
    */
    void BoostPickup::changedUsed(void)
    {
        SUPER(BoostPickup, changedUsed);

        SpaceShip* ship = this->carrierToSpaceShipHelper();
        if(ship == nullptr) // If the PickupCarrier is no SpaceShip, then this pickup is useless and therefore is destroyed.
            this->Pickupable::destroy();

        // If the pickup has transited to used.
        if(this->isUsed())
        {
            float gainedBoostPower = ship->getInitialBoostPower() * boostRefill_;
            ship->gainBoostPower(gainedBoostPower);

            // This will destroy the pickp
            this->setUsed(false);
        }
        else
        {
            this->Pickupable::destroy();
        }
    }

    /**
    @brief
        Helper to transform the PickupCarrier to a SpaceShip, and throw an error message if the conversion fails.
    @return
        A pointer to the SpaceShip, or nullptr if the conversion failed.
    */
    SpaceShip* BoostPickup::carrierToSpaceShipHelper(void)
    {
        PickupCarrier* carrier = this->getCarrier();
        SpaceShip* ship = orxonox_cast<SpaceShip*>(carrier);

        if(ship == nullptr)
        {
            orxout(internal_error, context::pickups) << "Invalid PickupCarrier in BoostPickup." << endl;
        }

        return ship;
    }


}

