/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Cyrill Burgener
 *
 */

/**
    @file HoverShip.cc
    @brief Implementation of the HoverShip control
*/

#include "HoverShip.h"
#include "core/CoreIncludes.h"
#include "core/XMLPort.h"

#include <BulletCollision/NarrowPhaseCollision/btManifoldPoint.h>

namespace orxonox
{
    RegisterClass(HoverShip);

    HoverShip::HoverShip(Context* context) : SpaceShip(context)
    {
        RegisterObject(HoverShip);
        enableCollisionCallback();
        isFloor_ = false;
        jumpBoost_ = 0;
    }

    void HoverShip::moveFrontBack(const Vector2& value)
        { this->steering_.z -= value.x; }

    void HoverShip::moveRightLeft(const Vector2& value)
        { this->steering_.x += value.x; }

    void HoverShip::moveUpDown(const Vector2& value)
        { this->steering_.y += value.x; }

    void HoverShip::rotateYaw(const Vector2& value)
    {
        this->localAngularAcceleration_.setY(this->localAngularAcceleration_.y() + value.x);

        Pawn::rotateYaw(value);
    }

    void HoverShip::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(HoverShip, XMLPort, xmlelement, mode);

        XMLPortParam(HoverShip, "jumpBoost", setJumpBoost, getJumpBoost, xmlelement, mode);
    }

    /**
    @brief
        Removed, does nothing.
    @param value
    */
    void HoverShip::rotatePitch(const Vector2& value) { }

    /**
    @brief
        Removed, does nothing.
    @param value
    */
    void HoverShip::rotateRoll(const Vector2& value) { }

    /**
    @brief
        Checks if the ship is touching the floor. The ship can only jump if there is contact with someting beneath it.
    */
    bool HoverShip::collidesAgainst(WorldEntity* otherObject, const btCollisionShape* cs, btManifoldPoint& contactPoint)
    {
        SpaceShip::collidesAgainst(otherObject, cs, contactPoint);
        //SUPER(HoverShip, collidesAgainst, otherObject, cs, contactPoint);

        if (contactPoint.m_normalWorldOnB.y() > 0.6
            && this->getVelocity().y < 1) {
            this->isFloor_ = true;
        } else {
            this->isFloor_ = false;
        }

        return false;
    }

    /**
    @brief
        Makes the ship jump
    @param bBoost
    */
    void HoverShip::boost(bool bBoost) {
        if (bBoost && this->isFloor_)
        {
            this->setVelocity(
                this->getVelocity().x,
                jumpBoost_,
                this->getVelocity().z
                );
            this->isFloor_ = false;
        }
    }
}
