/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file IceGun.cc
    @brief Implementation of the IceGun class.
*/

#include "IceGun.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"    
#include "weaponsystem/Weapon.h"
#include "weaponsystem/WeaponPack.h"
#include "weaponsystem/WeaponSystem.h"
#include "worldentities/pawns/Pawn.h"

#include "weapons/projectiles/IceGunProjectile.h"

namespace orxonox
{
    RegisterClass(IceGun);

    IceGun::IceGun(Context* context) : WeaponMode(context)
    {
        RegisterObject(IceGun);

        // Default values
        this->reloadTime_ = 1.0f;
        this->damage_ = 0.0f;
        this->speed_ = 1200.0f;

        this->setFreezeTime(3.0);
        this->setFreezeFactor(0.5);

        this->setMunitionName("IceMunition");
        this->setFireSound("sounds/Weapon_LightningGun.ogg");
        this->setReloadSound("sounds/Reload_IceGun.ogg", 0.4);

        hudImageString_ = "Orxonox/WSHUD_WM_IceGun";
    }

    IceGun::~IceGun()
    {
    }

    /**
    @brief
        XMLPort for the IceGun. You can define how often the projectiles split, how many childs should be created per split, the spread and the time between two splits.
    */
    void IceGun::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(IceGun, XMLPort, xmlelement, mode);

        XMLPortParam(IceGun, "freezetime", setFreezeTime, getFreezeTime, xmlelement, mode);
        XMLPortParam(IceGun, "freezefactor", setFreezeFactor, getFreezeFactor, xmlelement, mode);
    }

    /**
    @brief
        Fires the weapon. Creates a projectile and fires it.
    */
    void IceGun::fire()
    {
        IceGunProjectile* projectile = new IceGunProjectile(this->getContext());

        this->computeMuzzleParameters(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn()->getAimPosition());
        projectile->setOrientation(this->getMuzzleOrientation());
        projectile->setPosition(this->getMuzzlePosition());
        projectile->setVelocity(this->getMuzzleDirection() * this->speed_);

        // Pass important information to the projectile: Freeze time and freeze factor
        projectile->setFreezeTime(getFreezeTime());
        projectile->setFreezeFactor(getFreezeFactor());

        projectile->setShooter(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn());
        projectile->setDamage(this->getDamage());
        projectile->setShieldDamage(this->getShieldDamage());
        projectile->setHealthDamage(this->getHealthDamage());
    }
}
