/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file FlameGun.cc
    @brief Implementation of the FlameGun class.
*/

#include "FlameGun.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "weaponsystem/Weapon.h"
#include "weaponsystem/WeaponPack.h"
#include "weaponsystem/WeaponSystem.h"
#include "worldentities/pawns/Pawn.h"

#include "weapons/projectiles/FlameGunProjectile.h"

namespace orxonox
{
    RegisterClass(FlameGun);

    const int FlameGun::PROJECTILES_PER_FIRE = 3;

    FlameGun::FlameGun(Context* context) : WeaponMode(context)
    {
        RegisterObject(FlameGun);

        this->reloadTime_ = 0.1f;
        this->damage_ = 0.0f;
        this->speed_ = 550.0f;
        this->lifetime_ = 1.0;
        this->sideAcceleration_ = 2000.0;

        this->setMunitionName("FlameMunition");
        this->setFireSound("sounds/Weapon_FlameGun.ogg");

        hudImageString_ = "Orxonox/WSHUD_WM_FlameGun";
    }

    FlameGun::~FlameGun()
    {
    }

    /**
    @brief
        XMLPort for the FlameGun. You can define how often the projectiles Flame, how many childs should be created per Flame, the spread and the time between two Flames.
    */
    void FlameGun::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(FlameGun, XMLPort, xmlelement, mode);

        XMLPortParam(FlameGun, "Lifetime", setLifetime, getLifetime, xmlelement, mode);
    }

    /**
    @brief
        Fires the weapon. Creates a projectile and fires it.
    */
    void FlameGun::fire()
    {
        this->computeMuzzleParameters(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn()->getAimPosition());

        Vector3 muzzleDirection = this->getMuzzleDirection();
        Vector3 directionOffset = muzzleDirection.perpendicular();
        directionOffset.normalise();
        Degree angle = Degree(rnd(0,360));
        directionOffset = Quaternion(angle, muzzleDirection.normalisedCopy()) * directionOffset;
        directionOffset.normalise();

        FlameGunProjectile* projectile = new FlameGunProjectile(this->getContext());

        projectile->setOrientation(this->getMuzzleOrientation());
        projectile->setPosition(this->getMuzzlePosition());
        projectile->setVelocity(muzzleDirection * this->speed_);

        projectile->setLifetime(getLifetime());

        projectile->setShooter(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn());
        projectile->setDamage(this->getDamage());
        projectile->setShieldDamage(this->getShieldDamage());
        projectile->setHealthDamage(this->getHealthDamage());

        angle = Degree(360.0/PROJECTILES_PER_FIRE);

        for (int i = 0; i < PROJECTILES_PER_FIRE; ++i)
        {
            FlameGunProjectile* projectile = new FlameGunProjectile(this->getContext());

            projectile->setOrientation(this->getMuzzleOrientation());
            projectile->setPosition(this->getMuzzlePosition());
            projectile->setVelocity(muzzleDirection * this->speed_);
            projectile->setAcceleration(directionOffset*sideAcceleration_);

            projectile->setLifetime(getLifetime());

            projectile->setShooter(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn());
            projectile->setDamage(this->getDamage());
            projectile->setShieldDamage(this->getShieldDamage());
            projectile->setHealthDamage(this->getHealthDamage());

            directionOffset = Quaternion(angle, muzzleDirection.normalisedCopy()) * directionOffset;
            directionOffset.normalise();
        }



        
    }
}
