/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      Damian 'Mozork' Frick
 *      Matthias Binder
 *
 */

/**
    @file LevelManager.cc
    @brief Implementation of the LevelManager singleton.
*/

#include "LevelManager.h"
#include "LevelStatus.h"

#include <map>

#include "core/singleton/ScopedSingletonIncludes.h"
#include "core/commandline/CommandLineIncludes.h"
#include "core/config/ConfigValueIncludes.h"
#include "core/CoreIncludes.h"
#include "core/ClassTreeMask.h"
#include "core/Loader.h"
#include "core/Resource.h"
#include "core/XMLFile.h"
#include "Level.h"
#include "PlayerManager.h"


namespace orxonox
{




    SetCommandLineArgument(level, "").shortcut("l").information("Default level file (overrides LevelManager::defaultLevelName_ configValue)");

    ManageScopedSingleton(LevelManager, ScopeID::ROOT, false);

    RegisterAbstractClass(LevelManager).inheritsFrom<Configurable>();

    /**
    @brief
        Constructor. Registers the object, sets config values and initializes variables.
    */
    LevelManager::LevelManager()
    {
        RegisterObject(LevelManager);
        this->setConfigValues();



        // check override
        if (!CommandLineParser::getArgument("level")->hasDefaultValue())
        {
            ModifyConfigValue(defaultLevelName_, tset, CommandLineParser::getValue("level").get<std::string>());
        }

        this->compileAvailableLevelList();
        this->nextIndex_ = 0;
        this->nextLevel_ = this->availableLevels_.begin();

        buildallLevelStatus();
    }


    LevelManager::~LevelManager()
    {
        // Delete all the LevelInfoItem objects because the LevelManager created them
        for (LevelInfoItem* info : availableLevels_)
            info->destroy();
        for(unsigned int i = 0; i<allLevelStatus_.size();++i)
        {
            allLevelStatus_[i]->destroy();
        }
    }

    /**
    @brief
        Set the config values for this object.
    */
    void LevelManager::setConfigValues()
    {

        SetConfigValue(defaultLevelName_, "missionOne.oxw")
            .description("Sets the pre selection of the level in the main menu.");
        SetConfigValue(lastWonMission_,  "")
            .description("The last finished mission of a campaign");
        SetConfigValue(campaignMissions_,  std::vector<std::string>())
            .description("The list of missions in the campaign");
    }


    /**
    @brief
        Request activity for the input Level.
        The Level will be added to the list of Levels whose activity is requested. The list is accessed in a FIFO manner.
        If the Level is the only Level in the list it will be immediately activated. If not it will be activated as soon as it reaches the front of the list.
    @param level
        A pointer to the Level whose activity is requested.
    */
    void LevelManager::requestActivity(Level* level)
    {
        assert( std::find(this->levels_.begin(), this->levels_.end(), level)==this->levels_.end() );
        // If the level is already in list.
        if( std::find(this->levels_.begin(), this->levels_.end(), level)!=this->levels_.end() )
            return;
        // If it isn't insert it at the back.
        this->levels_.push_back(level);
        // If it is the only level in the list activate it.
        if (this->levels_.size() == 1)
            this->activateNextLevel();
    }

    /**
    @brief
        Release activity for the input Level.
        Removes the Level from the list. If the Level was the one currently active, it is deactivated and the next Level in line is activated.
    @param level
        A pointer to the Level whose activity is to be released.
    */
    void LevelManager::releaseActivity(Level* level)
    {
        if (this->levels_.size() > 0)
        {
            // If the level is the active level in the front of the list.
            if (this->levels_.front() == level)
            {
                // Deactivate it, remove it from the list and activate the next level in line.
                level->setActive(false);
                this->levels_.pop_front();
                this->activateNextLevel();
            }
            else // Else just remove it from the list.
                this->levels_.erase(std::find(this->levels_.begin(), this->levels_.end(), level));
        }
    }

    /**
    @brief
        Get the currently active Level.
    @return
        Returns a pointer to the currently active level or nullptr if there currently are no active Levels.
    */
    Level* LevelManager::getActiveLevel()
    {
        if (this->levels_.size() > 0)
            return this->levels_.front();
        else
            return nullptr;
    }

    /**
    @brief
        Activate the next Level.
    */
    void LevelManager::activateNextLevel()
    {

        if (this->levels_.size() > 0)
        {
            // Activate the level that is the first in the list of levels whose activity has been requested.
            this->levels_.front()->setActive(true);
            // Make every player enter the newly activated level.
            for (const auto& mapEntry : PlayerManager::getInstance().getClients())
                this->levels_.front()->playerEntered(mapEntry.second);
        }
    }

    /**
    @brief
        Set the default Level.
    @param levelName
        The filename of the default Level.
    */
    void LevelManager::setDefaultLevel(const std::string& levelName)
    {
        ModifyConfigValue(defaultLevelName_, set, levelName);
    }

    /**
    @brief
        Get the number of available Levels.
        Also updates the list of available Levels.
    @return
        Returns the number of available Levels.
    */
    unsigned int LevelManager::getNumberOfLevels()
    {
        this->updateAvailableLevelList();

        return this->availableLevels_.size();
    }

    /**
    @brief
        Get the LevelInfoItem at the given index in the list of available Levels.
        The LevelInfoItems are sorted in alphabetical order accoridng to the name of the Level.
        This method is most efficiently called with consecutive indices (or at least ascending indices).
    @param index
        The index of the item that should be returned.
    @return
        Returns a pointer to the LevelInfoItem at the given index.
    */
    LevelInfoItem* LevelManager::getAvailableLevelListItem(unsigned int index)
    {
        if(index >= this->availableLevels_.size())
            return nullptr;

        // If this index directly follows the last we can optimize a lot.
        if(index == this->nextIndex_)
        {
            this->nextIndex_++;
            std::set<LevelInfoItem*, LevelInfoCompare>::iterator it = this->nextLevel_;
            this->nextLevel_++;
            return *it;
        }
        else
        {
            // If this index is bigger than the last, we can optimize a little.
            if(index < this->nextIndex_)
            {
                this->nextIndex_ = 0;
                this->nextLevel_ = this->availableLevels_.begin();
            }

            while(this->nextIndex_ != index)
            {
                this->nextIndex_++;
                this->nextLevel_++;
            }
            this->nextIndex_++;
            std::set<LevelInfoItem*, LevelInfoCompare>::iterator it = this->nextLevel_;
            this->nextLevel_++;
            return *it;
        }
    }

    /**
    @brief
        Compile the list of available Levels.
        Iterates over all *.oxw files, loads the LevelInfo objects in them and from that it creates the LevelInfoItems which are inserted in a list.
    */
    void LevelManager::compileAvailableLevelList()
    {
        // Get all files matching the level criteria
        Ogre::StringVectorPtr levels = Resource::findResourceNames("*.oxw");

        // We only want to load as little as possible
        ClassTreeMask mask;
        mask.exclude(Class(BaseObject));
        mask.include(Class(LevelInfo));

        // Iterate over all the found *.oxw files
        orxout(internal_info) << "Loading LevelInfos..." << endl;
        std::set<std::string> names;
        for (Ogre::StringVector::const_iterator it = levels->begin(); it != levels->end(); ++it)
        {
            // TODO: Replace with tag?
            if (it->find("old/") != 0)
            {
                LevelInfoItem* info = nullptr;

                // Load the LevelInfo object from the level file.
                XMLFile file = XMLFile(*it);
                Loader::getInstance().load(&file, mask, false, true);

                // Find the LevelInfo object we've just loaded (if there was one)
                for(LevelInfo* levelInfo : ObjectList<LevelInfo>())
                    if(levelInfo->getXMLFilename() == *it)
                        info = levelInfo->copy();

                // We don't need the loaded stuff anymore
                Loader::getInstance().unload(&file);

                if(info == nullptr)
                {
                    // Create a default LevelInfoItem object that merely contains the name
                    std::string filenameWOExtension = it->substr(0, it->find(".oxw"));
                    info = new LevelInfoItem(filenameWOExtension, *it);
                }

                // Warn about levels with the same name.
                if(!names.insert(info->getName()).second)
                    orxout(internal_warning) << "Multiple levels (" << info->getXMLFilename() << ") with name '" << info->getName() << "' found!" << endl;

                // Warn about multiple items so that it gets fixed quickly
                if(availableLevels_.find(info) != availableLevels_.end())
                {
                    orxout(internal_warning) << "Multiple levels (" << info->getXMLFilename() << ") with same name '" << info->getName() << "' and filename found! Exluding..." << endl;
                    // Delete LevelInfoItem to avoid a dangling pointer
                    delete info;
                }
                else
                    this->availableLevels_.insert(info);
            }
        }
    }

    /**
    @brief
        Update the list of available Levels.
    */
    void LevelManager::updateAvailableLevelList(void)
    {
        //TODO: Implement some kind of update?
    }



    /**
    @brief
        first updates allLevelStatus and then check if the level with index i is activated
    */
    int LevelManager::missionactivate(int index)
    {
        updateAllLevelStatus();
        int activated = allLevelStatus_[index]->activated;
        return activated;
    }
    /**
    @brief
        update the last won mission to won=true
    */

    void LevelManager::updatewon(int lastwon)
    {
        allLevelStatus_[lastwon]->won=true;

    }

    /**
    @brief
        checks if a level is won
        if a level is won, the other levels should be shown as saved in 'nextLevels' of the won level
        therefore 'activated' of all other levels will be updated to the correct value
        if more than one level is won, the level with the highes index decides which other levels will be shown, activated or not activated
    */

    void LevelManager::updateAllLevelStatus()
    {
        for(unsigned int i =0;i<allLevelStatus_.size();i++)
        {
            if(allLevelStatus_[i]->won)
            {
                allLevelStatus_[i]->activated=1;
                std::vector<int> nextLevels=allLevelStatus_[i]->nextLevels;
                for(unsigned int j=0;j<allLevelStatus_.size();j++)
                {
                    allLevelStatus_[j]->activated=nextLevels[j];
                }
            }
        }
    }

    /**
    @brief
        the name of the last won mission is saved in the config file
    */

    void LevelManager::setLevelStatus(const std::string& LevelWon)
    {
        
            
        ModifyConfigValue(lastWonMission_, set, LevelWon);

        /**
        TODO: save allLevelWon_ into the config file
        */
    }


    /**
    @brief
        build up allLevelStatus_
        has to be done once per game (not per level)
        all connections between the levels are saved in the allLevelStatus_
        also the won variable of the LevelStatus have to be set to the corresponding allLevelWon_ values
    */
    void LevelManager::buildallLevelStatus()
    {
        for(unsigned int i =0;i<campaignMissions_.size();++i)
        {
            LevelStatus* level= new LevelStatus(this->getContext());
            allLevelStatus_.push_back(level);
        }

        /**
        TODO: allLevelWon_ should be loaded into the allLevelSatus_ such that the progress of the campaign can be saved
        */

        allLevelStatus_[0]->activated=1;


        std::vector<int> v={1,1,1,0,0,0,0,0,0};
        allLevelStatus_[0]->nextLevels=v;

        v={1,1,2,1,0,0,0,0,0};
        allLevelStatus_[1]->nextLevels=v;

        v={1,2,1,0,1,0,0,0,0};
        allLevelStatus_[2]->nextLevels=v;

        v={1,1,2,1,0,1,1,0,0};
        allLevelStatus_[3]->nextLevels=v;

        v={1,2,1,0,1,0,0,1,1};
        allLevelStatus_[4]->nextLevels=v;

        v={1,1,2,1,0,1,2,0,0};
        allLevelStatus_[5]->nextLevels=v;

        v={1,1,2,1,0,2,1,0,0};
        allLevelStatus_[6]->nextLevels=v;

        v={1,2,1,0,1,0,0,1,2};
        allLevelStatus_[7]->nextLevels=v;

        v={1,2,1,0,1,0,0,2,1};
        allLevelStatus_[8]->nextLevels=v;

    }
}
