/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Manuel Meier
 *   Co-authors:
 *      Cyrill Burgener
 *      Tomer Gidron
 *
 */

/**
    @file Hover.cc
    @brief Implementation of the Hover class. Sets up the whole Minigame
*/

#include "Hover.h"
#include "chat/ChatManager.h"
#include "HoverOrigin.h"
#include "HoverWall.h"
#include "HoverFlag.h"
#include "MazeGenerator.h"
#include "core/CoreIncludes.h"
#include "gamestates/GSLevel.h"
#include "HoverShip.h"

#include "pickup/PickupSpawner.h"
#include "pickup/Pickup.h"

namespace orxonox
{
    RegisterUnloadableClass(Hover);

    Hover::Hover(Context* context) : Gametype(context)
    {
        RegisterObject(Hover);

        this->origin_ = nullptr;
        this->numberOfFlags_ = 1;
        this->firstTick_ = true;
        level = 1; //start at level 1
        flagsTaken = 0;// took 0 flags in the beginning
        lives = 3;

        numCells = 0;
        cellSize = 0;
        cellHeight = 0;

        bLevelUpgradeHUD = false;

        totFlags = 0;

        this->setHUDTemplate("HoverHUD");

    }

    void Hover::start()
    {
        Gametype::start();
        if(this->firstTick_ && this->origin_)
        {
            this->firstTick_ = false;

            numCells = this->origin_->getNumCells();
            cellSize = this->origin_->getCellSize();
            cellHeight = this->origin_->getCellHeight();

            //TODO
            //create boolean array/vector to assert that no two objects are placed in the same way
            

            MazeGenerator generator(numCells);
            generator.generateMaze();
            generator.renderMaze();

            int* levelcode = generator.getLevelcode();

            //Outer Walls
            for(int i = 0; i<numCells; i++){
                (new HoverWall(origin_->getContext()))->init(0,        i+1,      cellSize, cellHeight, 1);
                (new HoverWall(origin_->getContext()))->init(numCells, i+1,      cellSize, cellHeight, 1);
                (new HoverWall(origin_->getContext()))->init(i+1,      0,        cellSize, cellHeight, 2);
                (new HoverWall(origin_->getContext()))->init(i+1,      numCells, cellSize, cellHeight, 2);
            }


            //Ground
            for(int i = 0; i<numCells; i++){
                for(int j = 0; j<numCells; j++){
                    StaticEntity* groundCell = new StaticEntity(origin_->getContext());

                    groundCell->addTemplate(origin_->getGroundTemplate());
                    groundCell->setPosition(get3dCoordinates(i,j,-60));
                }

            }



            //Generate inner Walls according to levelcode
            for(int y=0; y<numCells; y++){
                for(int x=0; x<numCells; x++){
                    switch(levelcode[ y * numCells + x ])
                    {
                        case 1: (new HoverWall(origin_->getContext()))->init(x+1, numCells-y, cellSize, cellHeight, 1);
                            break;
                        case 3: (new HoverWall(origin_->getContext()))->init(x+1, numCells-y, cellSize, cellHeight, 1);
                        case 2: (new HoverWall(origin_->getContext()))->init(x+1, numCells-y, cellSize, cellHeight, 0);
                        default:
                            break;
                    }
                }
            }

            createFlags();

            //Generate 3 PickupSpawners randomly (destroy hover pickup)
            for (int i = 0; i<3; i++)
            {
                PickupSpawner* pickupSpawner = new PickupSpawner(origin_->getContext());

                pickupSpawner->setPosition(get3dCoordinates(rand()%numCells, rand()%numCells, 0.0f));
                pickupSpawner->setPickupTemplateName(origin_->getPickupTemplate());
                pickupSpawner->setMaxSpawnedItems(3);
                pickupSpawner->setRespawnTime(30);
                pickupSpawner->setTriggerDistance(40);
                // Add pickup spawner to the pickup spawner list
                pickupSpawners_.push_back(pickupSpawner);
            }
            
            //Generate 3 PickupSpawners randomly (speed pickup)
            for (int i = 0; i<3; i++)
            {
                PickupSpawner* pickupSpawner = new PickupSpawner(origin_->getContext());

                pickupSpawner->setPosition(get3dCoordinates(rand()%numCells, rand()%numCells, 0.0f));
                pickupSpawner->setPickupTemplateName(origin_->getPickupTemplateSpeed());
                pickupSpawner->setMaxSpawnedItems(3);
                pickupSpawner->setRespawnTime(30);
                pickupSpawner->setTriggerDistance(40);
                // Add pickup spawner to the pickup spawner list
                pickupSpawners_.push_back(pickupSpawner);
            }

            //Generate 3 PickupSpawners randomly (shrink pickup)
            for (int i = 0; i<3; i++)
            {
                PickupSpawner* pickupSpawner = new PickupSpawner(origin_->getContext());

                pickupSpawner->setPosition(get3dCoordinates(rand()%numCells, rand()%numCells, 0.0f));
                pickupSpawner->setPickupTemplateName(origin_->getPickupTemplateShrink());
                pickupSpawner->setMaxSpawnedItems(3);
                pickupSpawner->setRespawnTime(30);
                pickupSpawner->setTriggerDistance(40);
                // Add pickup spawner to the pickup spawner list
                pickupSpawners_.push_back(pickupSpawner);
            }

            //*****************************************************************************

            //Generate destroyable crates randomly on field

            for (int i = 0; i<10; i++){

                Pawn* crate = new Pawn(origin_->getContext());

                crate->addTemplate(origin_->getObstacleTemplate());  
                crate->setPosition(get3dCoordinates(rand()%numCells, rand()%numCells, 43.0f));


            }    
            



             //If no lives are left, end game
            if(lives <= 0)
            {
                GSLevel::startMainMenu();
            }
            
        }
    }


    // generate new Flags
    void Hover::createFlags()
    {
        //TODO
        //Generate flags randomly using bool array

        //Generate 5 flags randomly on field
        for ( int i = 0; i < 5; i++ )
        {
            HoverFlag* flag = new HoverFlag(origin_->getContext());
            flag->init(rand()%numCells, rand()%numCells, cellSize);
            flags_.push_back(flag);

            

            if(flags_[i]->getPosition() == get3dCoordinates(0,0,-60))
            {
                flags_[i]->destroyLater();
                flags_.erase(flags_.begin()+i);
            }
        }
            
    }

    void Hover::tick(float dt)
    {
        SUPER(Hover, tick, dt);


        // Check if ship collided with one of the flags
        for ( unsigned int i = 0; i < flags_.size(); i++ )
        {
            if(flags_[i]->getCollided())
            {
                flags_[i]->destroyLater();
                flags_.erase (flags_.begin()+i);
                totFlags++;
                if(flags_.size()<=0){
                    //ChatManager::message("Level Up!");
                    
                    levelUp();
                }
            }

        }
        numberOfFlags_ = flags_.size();

        if(lives <= 0){
                GSLevel::startMainMenu();
            }
    }

    //start new level - call to create new flags
    void Hover::levelUp()
    {
        level++;
        //increment lives after every 4 levels
        if(level%4 == 0)
        {
            lives++;
        }
        createFlags();
        toggleShowLevel();
        showLevelTimer.setTimer(2.0f, false, createExecutor(createFunctor(&Hover::toggleShowLevel, this)));

        //spawn one additional crate randomly
        Pawn* crate = new Pawn(origin_->getContext());
        crate->addTemplate(origin_->getObstacleTemplate());  
        crate->setPosition(get3dCoordinates(rand()%numCells, rand()%numCells, 43.0f));

    }

    Vector3 Hover::get3dCoordinates(int x, int y, float heightOffset)
    {
        return Vector3(x*cellSize*1.0f + cellSize/2, heightOffset, y*cellSize*1.0f + cellSize/2);
    }

    //if killed, subtract number of lives. If lives == 0, end game
    void Hover::costLife()
    {
        lives--;
        if (lives <= 0)
            GSLevel::startMainMenu();
    }
}
