/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

#include "Plugin.h"

#include "ModuleInstance.h"
#include "util/Output.h"
#include "core/Core.h"
#include "StaticInitializationManager.h"

namespace orxonox
{
    Plugin::Plugin(const std::string& name, const std::string& libraryName) : name_(name), libraryName_(libraryName)
    {
        this->referenceCounter_ = 0;
        this->moduleInstance_ = nullptr;
    }

    Plugin::~Plugin()
    {
        // force unloading of the module when the plugin is destroyed
        if (this->moduleInstance_ != nullptr)
            this->unloadModule();
    }

    void Plugin::reference()
    {
        this->referenceCounter_++;
        if (this->referenceCounter_ == 1) // increased from 0 to 1 -> load plugin
            this->load();
        else
            orxout(internal_info) << "Increased reference count for plugin " << this->name_ << " to " << this->referenceCounter_ << endl;
    }

    void Plugin::dereference(bool bMerelyDeactivate)
    {
        if (this->referenceCounter_ == 0)
        {
            orxout(internal_warning) << "Tried to dereference plugin " << this->name_ << " more often than it was referenced" << endl;
            return;
        }

        this->referenceCounter_--;
        if (this->referenceCounter_ == 0) // reduced from 1 to 0 -> load plugin
            this->unload(bMerelyDeactivate);
        else
            orxout(internal_info) << "Reduced reference count for plugin " << this->name_ << " to " << this->referenceCounter_ << endl;
    }

    //////////////////////////////////////////
    //                 LOAD                 //
    //////////////////////////////////////////
    void Plugin::load()
    {
        // only load module if it isn't already loaded. otherwise merely activate it.
        if (this->moduleInstance_ == nullptr)
            this->loadModule();
        else
            this->activateModule();
    }
    void Plugin::loadModule()
    {
        orxout(internal_info) << "Loading plugin " << this->name_ << "..." << endl;
        this->moduleInstance_ = new ModuleInstance(this->libraryName_);
        Core::getInstance().loadModule(this->moduleInstance_);
    }
    void Plugin::activateModule()
    {
        orxout(internal_info) << "Activating plugin " << this->name_ << "..." << endl;
        StaticInitializationManager::getInstance().loadModule(this->moduleInstance_);
    }

    //////////////////////////////////////////
    //                UNLOAD                //
    //////////////////////////////////////////
    void Plugin::unload(bool bMerelyDeactivate)
    {
        // fully unload the module unless otherwise requested.
        if (bMerelyDeactivate)
            this->deactivateModule();
        else
            this->unloadModule();
    }
    void Plugin::unloadModule()
    {
        orxout(internal_info) << "Unloading plugin " << this->name_ << "..." << endl;
        Core::getInstance().unloadModule(this->moduleInstance_);
        delete this->moduleInstance_;
        this->moduleInstance_ = nullptr;
    }
    void Plugin::deactivateModule()
    {
        orxout(internal_info) << "Deactivating plugin " << this->name_ << "..." << endl;
        StaticInitializationManager::getInstance().unloadModule(this->moduleInstance_);
    }
}
