/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      Simon Miescher
 *
 */

#include "Pawn.h"

#include <algorithm>

#include "core/CoreIncludes.h"
#include "core/GameMode.h"
#include "core/XMLPort.h"
#include "core/EventIncludes.h"
#include "network/NetworkFunction.h"

#include "infos/PlayerInfo.h"
#include "controllers/Controller.h"
#include "gametypes/Gametype.h"
#include "graphics/ParticleSpawner.h"
#include "worldentities/ExplosionChunk.h"
#include "worldentities/ExplosionPart.h"
#include "weaponsystem/WeaponSystem.h"
#include "weaponsystem/WeaponSlot.h"
#include "weaponsystem/WeaponPack.h"
#include "weaponsystem/WeaponSet.h"
#include "weaponsystem/Munition.h"
#include "sound/WorldSound.h"

#include "controllers/FormationController.h"

namespace orxonox
{
    RegisterClass(Pawn);

    Pawn::Pawn(Context* context)
        : ControllableEntity(context)
        , RadarViewable(this, static_cast<WorldEntity*>(this))
    {
        RegisterObject(Pawn);

        this->bAlive_ = true;
        this->bVulnerable_ = true;

        this->health_ = 0;
        this->maxHealth_ = 0;
        this->initialHealth_ = 0;

        this->shieldHealth_ = 0;
        this->initialShieldHealth_ = 0;
        this->maxShieldHealth_ = 100; //otherwise shield might increase to float_max
        this->shieldAbsorption_ = 0.5;
        this->shieldRechargeRate_ = 0;
        this->shieldRechargeWaitTime_ = 1.0f;
        this->shieldRechargeWaitCountdown_ = 0;

        this->lastHitOriginator_ = nullptr;

        // set damage multiplier to default value 1, meaning nominal damage
        this->damageMultiplier_ = 1;

        this->spawnparticleduration_ = 3.0f;

        this->aimPosition_ = Vector3::ZERO;

        //this->explosionPartList_ = nullptr;

        if (GameMode::isMaster())
        {
            this->weaponSystem_ = new WeaponSystem(this->getContext());
            this->weaponSystem_->setPawn(this);
        }
        else
            this->weaponSystem_ = nullptr;

        this->setRadarObjectColour(ColourValue::Red);
        this->setRadarObjectShape(RadarViewable::Shape::Dot);

        this->registerVariables();

        this->isHumanShip_ = this->hasLocalController();

        this->setSyncMode(ObjectDirection::Bidirectional); // needed to synchronise e.g. aimposition

        if (GameMode::isMaster())
        {
            this->explosionSound_ = new WorldSound(this->getContext());
            this->explosionSound_->setVolume(1.0f);
        }
        else
        {
            this->explosionSound_ = nullptr;
        }
    }

    Pawn::~Pawn()
    {
        if (this->isInitialized())
        {
            if (this->weaponSystem_)
                this->weaponSystem_->destroy();
        }
    }

    void Pawn::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(Pawn, XMLPort, xmlelement, mode);

        XMLPortParam(Pawn, "health", setHealth, getHealth, xmlelement, mode).defaultValues(100);
        XMLPortParam(Pawn, "maxhealth", setMaxHealth, getMaxHealth, xmlelement, mode).defaultValues(200);
        XMLPortParam(Pawn, "initialhealth", setInitialHealth, getInitialHealth, xmlelement, mode).defaultValues(100);

        XMLPortParam(Pawn, "shieldhealth", setShieldHealth, getShieldHealth, xmlelement, mode).defaultValues(0);
        XMLPortParam(Pawn, "initialshieldhealth", setInitialShieldHealth, getInitialShieldHealth, xmlelement, mode).defaultValues(0);
        XMLPortParam(Pawn, "maxshieldhealth", setMaxShieldHealth, getMaxShieldHealth, xmlelement, mode).defaultValues(100);
        XMLPortParam(Pawn, "shieldabsorption", setShieldAbsorption, getShieldAbsorption, xmlelement, mode).defaultValues(0);

        XMLPortParam(Pawn, "vulnerable", setVulnerable, isVulnerable, xmlelement, mode).defaultValues(true);

        XMLPortParam(Pawn, "spawnparticlesource", setSpawnParticleSource, getSpawnParticleSource, xmlelement, mode);
        XMLPortParam(Pawn, "spawnparticleduration", setSpawnParticleDuration, getSpawnParticleDuration, xmlelement, mode).defaultValues(3.0f);
        XMLPortParam(Pawn, "explosionchunks", setExplosionChunks, getExplosionChunks, xmlelement, mode).defaultValues(0);

        XMLPortObject(Pawn, WeaponSlot, "weaponslots", addWeaponSlot, getWeaponSlot, xmlelement, mode);
        XMLPortObject(Pawn, WeaponSet, "weaponsets", addWeaponSet, getWeaponSet, xmlelement, mode);
        XMLPortObject(Pawn, WeaponPack, "weaponpacks", addWeaponPackXML, getWeaponPack, xmlelement, mode);
        XMLPortObject(Pawn, Munition, "munition", addMunitionXML, getMunitionXML, xmlelement, mode);

        XMLPortObject(Pawn, ExplosionPart, "explosion", addExplosionPart, getExplosionPart, xmlelement, mode);
        XMLPortParam(Pawn, "shieldrechargerate", setShieldRechargeRate, getShieldRechargeRate, xmlelement, mode).defaultValues(0);
        XMLPortParam(Pawn, "shieldrechargewaittime", setShieldRechargeWaitTime, getShieldRechargeWaitTime, xmlelement, mode).defaultValues(1.0f);

        XMLPortParam(Pawn, "explosionSound",  setExplosionSound,  getExplosionSound,  xmlelement, mode);

        XMLPortParam ( RadarViewable, "radarname", setRadarName, getRadarName, xmlelement, mode );
    }

    void Pawn::XMLEventPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(Pawn, XMLEventPort, xmlelement, mode);

        XMLPortEventState(Pawn, BaseObject, "vulnerability", setVulnerable, xmlelement, mode);
    }

    void Pawn::registerVariables()
    {
        registerVariable(this->bAlive_,            VariableDirection::ToClient);
        registerVariable(this->health_,            VariableDirection::ToClient);
        registerVariable(this->maxHealth_,         VariableDirection::ToClient);
        registerVariable(this->shieldHealth_,      VariableDirection::ToClient);
        registerVariable(this->maxShieldHealth_,   VariableDirection::ToClient);
        registerVariable(this->shieldAbsorption_,  VariableDirection::ToClient);
        registerVariable(this->aimPosition_,       VariableDirection::ToServer);  // For the moment this variable gets only transfered to the server
    }

    void Pawn::tick(float dt)
    {
        //BigExplosion* chunk = new BigExplosion(this->getContext());
        SUPER(Pawn, tick, dt);

        // Recharge the shield
        // TODO: use the existing timer functions instead
        if(this->shieldRechargeWaitCountdown_ > 0)
        {
            this->decreaseShieldRechargeCountdownTime(dt);
        }
        else
        {
            this->addShieldHealth(this->getShieldRechargeRate() * dt);
            this->resetShieldRechargeCountdown();
        }

        if (GameMode::isMaster())
        {
            if (this->health_ <= 0 && bAlive_)
            {
                this->fireEvent(); // Event to notify anyone who wants to know about the death.
                this->death();
            }
        }
    }

    void Pawn::preDestroy()
    {
        // yay, multiple inheritance!
        this->ControllableEntity::preDestroy();
        this->PickupCarrier::preDestroy();
    }

    void Pawn::setPlayer(PlayerInfo* player)
    {
        ControllableEntity::setPlayer(player);

        if (this->getGametype())
            this->getGametype()->playerStartsControllingPawn(player, this);
    }

    void Pawn::removePlayer()
    {
        if (this->getGametype())
            this->getGametype()->playerStopsControllingPawn(this->getPlayer(), this);

        ControllableEntity::removePlayer();
    }


    void Pawn::setHealth(float health)
    {
        this->health_ = std::min(health, this->maxHealth_); //Health can't be set to a value bigger than maxHealth, otherwise it will be reduced at first hit
    }

    void Pawn::setShieldHealth(float shieldHealth)
    {
        this->shieldHealth_ = std::min(shieldHealth, this->maxShieldHealth_);
    }

    void Pawn::setMaxShieldHealth(float maxshieldhealth)
    {
        this->maxShieldHealth_ = maxshieldhealth;
    }

    void Pawn::setShieldRechargeRate(float shieldRechargeRate)
    {
        this->shieldRechargeRate_ = shieldRechargeRate;
    }

    void Pawn::setShieldRechargeWaitTime(float shieldRechargeWaitTime)
    {
        this->shieldRechargeWaitTime_ = shieldRechargeWaitTime;
    }

    void Pawn::decreaseShieldRechargeCountdownTime(float dt)
    {
        this->shieldRechargeWaitCountdown_ -= dt;
    }

    void Pawn::changedVulnerability()
    {

    }

    void Pawn::damage(float damage, float healthdamage, float shielddamage, Pawn* originator, const btCollisionShape* cs)
    {
        // A pawn can only get damaged if it is vulnerable
        if (!isVulnerable())
        {
            return;
        }

        // Applies multiplier given by the DamageBoost Pickup.
        if (originator)
            damage *= originator->getDamageMultiplier();

        if (this->getGametype() && this->getGametype()->allowPawnDamage(this, originator))
        {
            // Health-damage cannot be absorbed by shields.
            // Shield-damage only reduces shield health. 
            // Normal damage can be (partially) absorbed by shields.

            if (shielddamage >= this->getShieldHealth())
            {
                this->setShieldHealth(0);
                this->setHealth(this->health_ - (healthdamage + damage));
            }
            else
            {
                this->setShieldHealth(this->shieldHealth_ - shielddamage);

                // remove remaining shieldAbsorpton-Part of damage from shield
                shielddamage = damage * this->shieldAbsorption_;
                shielddamage = std::min(this->getShieldHealth(),shielddamage);
                this->setShieldHealth(this->shieldHealth_ - shielddamage);

                // set remaining damage to health
                this->setHealth(this->health_ - (damage - shielddamage) - healthdamage);
            }

            this->lastHitOriginator_ = originator;
        }
    }

// TODO: Still valid?
/* HIT-Funktionen
    Die hit-Funktionen muessen auch in src/orxonox/controllers/Controller.h angepasst werden! (Visuelle Effekte)

*/
    void Pawn::hit(Pawn* originator, const Vector3& force, const btCollisionShape* cs, float damage, float healthdamage, float shielddamage)
    {
        if (this->getGametype() && this->getGametype()->allowPawnHit(this, originator) && (!this->getController() || !this->getController()->getGodMode()) )
        {
            this->damage(damage, healthdamage, shielddamage, originator, cs);
            this->setVelocity(this->getVelocity() + force);
        }
    }

    void Pawn::hit(Pawn* originator, btManifoldPoint& contactpoint, const btCollisionShape* cs, float damage, float healthdamage, float shielddamage)
    {
        if (this->getGametype() && this->getGametype()->allowPawnHit(this, originator) && (!this->getController() || !this->getController()->getGodMode()) )
        {
            this->damage(damage, healthdamage, shielddamage, originator, cs);

            if ( this->getController() )
                this->getController()->hit(originator, contactpoint, damage); // changed to damage, why shielddamage?
        }
    }

    void Pawn::kill()
    {
        this->damage(this->health_);
        this->death();
    }

    void Pawn::spawneffect()
    {
        // play spawn effect
        if (!this->spawnparticlesource_.empty())
        {
            ParticleSpawner* effect = new ParticleSpawner(this->getContext());
            effect->setPosition(this->getPosition());
            effect->setOrientation(this->getOrientation());
            effect->setDestroyAfterLife(true);
            effect->setSource(this->spawnparticlesource_);
            effect->setLifetime(this->spawnparticleduration_);
        }
    }

    void Pawn::death()
    {
        this->setHealth(1);
        if (this->getGametype() && this->getGametype()->allowPawnDeath(this, this->lastHitOriginator_))
        {
            // Set bAlive_ to false and wait for destroyLater() to do the destruction
            this->bAlive_ = false;
            this->destroyLater();

            this->setDestroyWhenPlayerLeft(false);

            if (this->getGametype())
                this->getGametype()->pawnKilled(this, this->lastHitOriginator_);

            if (this->getPlayer() && this->getPlayer()->getControllableEntity() == this)
            {
                // Start to control a new entity if you're the master of a formation
                if(this->hasSlaves())
                {
                    Controller* slave = this->getSlave();
                    ControllableEntity* entity = slave->getControllableEntity();


                    if(!entity->hasHumanController())
                    {
                        // delete the AIController // <-- TODO: delete? nothing is deleted here... should we delete the controller?
                        slave->setControllableEntity(nullptr);

                        // set a new master within the formation
                        orxonox_cast<FormationController*>(this->getController())->setNewMasterWithinFormation(orxonox_cast<FormationController*>(slave));

                        // start to control a slave
                        this->getPlayer()->startControl(entity);
                    }
                    else
                    {
                        this->getPlayer()->stopControl();
                    }

                }
                else
                {
                    this->getPlayer()->stopControl();
                }
            }
            if (GameMode::isMaster())
            {
                this->goWithStyle();
            }
        }
    }
    void Pawn::goWithStyle()
    {

        this->bAlive_ = false;
        this->setDestroyWhenPlayerLeft(false);

        while(!explosionPartList_.empty())
        {
            explosionPartList_.back()->setPosition(this->getPosition());
            explosionPartList_.back()->setVelocity(this->getVelocity());
            explosionPartList_.back()->setOrientation(this->getOrientation());
            explosionPartList_.back()->Explode();
            explosionPartList_.pop_back();
        }

        for (unsigned int i = 0; i < this->numexplosionchunks_; ++i)
        {
            ExplosionChunk* chunk = new ExplosionChunk(this->getContext());
            chunk->setPosition(this->getPosition());
        }

        this->explosionSound_->setPosition(this->getPosition());
        this->explosionSound_->play();
    }

    /**
    @brief
        Check whether the Pawn has a @ref Orxonox::WeaponSystem and fire it with the specified firemode if it has one.
    */

    void Pawn::fired(unsigned int firemode)
    {
        if (this->weaponSystem_)
            this->weaponSystem_->fire(firemode);
    }

    void Pawn::postSpawn()
    {
        this->setHealth(this->initialHealth_);
        if (GameMode::isMaster())
            this->spawneffect();
    }


    void Pawn::addExplosionPart(ExplosionPart* ePart)
    {this->explosionPartList_.push_back(ePart);}


    ExplosionPart * Pawn::getExplosionPart()
    {return this->explosionPartList_.back();}



    /* WeaponSystem:
    *   functions load Slot, Set, Pack from XML and make sure all parent-pointers are set.
    *   with setWeaponPack you can not just load a Pack from XML but if a Pack already exists anywhere, you can attach it.
    *       --> e.g. Pickup-Items
    */
    void Pawn::addWeaponSlot(WeaponSlot * wSlot)
    {
        this->attach(wSlot);
        if (this->weaponSystem_)
            this->weaponSystem_->addWeaponSlot(wSlot);
    }

    WeaponSlot * Pawn::getWeaponSlot(unsigned int index) const
    {
        if (this->weaponSystem_)
            return this->weaponSystem_->getWeaponSlot(index);
        else
            return nullptr;
    }

    void Pawn::addWeaponSet(WeaponSet * wSet)
    {
        if (this->weaponSystem_)
            this->weaponSystem_->addWeaponSet(wSet);
    }

    WeaponSet * Pawn::getWeaponSet(unsigned int index) const
    {
        if (this->weaponSystem_)
            return this->weaponSystem_->getWeaponSet(index);
        else
            return nullptr;
    }

    void Pawn::addWeaponPack(WeaponPack * wPack)
    {
        if (this->weaponSystem_)
        {
            this->weaponSystem_->addWeaponPack(wPack);
            this->addedWeaponPack(wPack);
        }
    }

    void Pawn::addWeaponPackXML(WeaponPack * wPack)
    {
        if (this->weaponSystem_)
        {
            if (!this->weaponSystem_->addWeaponPack(wPack))
                wPack->destroy();
            else
                this->addedWeaponPack(wPack);
        }
    }

    WeaponPack * Pawn::getWeaponPack(unsigned int index) const
    {
        if (this->weaponSystem_)
            return this->weaponSystem_->getWeaponPack(index);
        else
            return nullptr;
    }

    void Pawn::addMunitionXML(Munition* munition)
    {
        if (this->weaponSystem_ && munition)
        {
            this->weaponSystem_->addMunition(munition);
        }
    }

    Munition* Pawn::getMunitionXML() const
    {
        return nullptr;
    }

    Munition* Pawn::getMunition(SubclassIdentifier<Munition> * identifier)
    {
        if (weaponSystem_)
        {
            return weaponSystem_->getMunition(identifier);
        }

        return nullptr;
    }

    //Tell the Map (RadarViewable), if this is a playership
    void Pawn::startLocalHumanControl()
    {
//        SUPER(ControllableEntity, startLocalHumanControl());
        ControllableEntity::startLocalHumanControl();
        this->isHumanShip_ = true;
    }

    void Pawn::changedVisibility(void)
    {
        SUPER(Pawn, changedVisibility);

        // enable proper radarviewability when the visibility is changed
        this->RadarViewable::settingsChanged();
    }


    // A function to check if this pawn's controller is the master of any formationcontroller
    bool Pawn::hasSlaves()
    {
        for (FormationController* controller : ObjectList<FormationController>())
        {
            // checks if the pawn's controller has a slave
            if (this->hasHumanController() && controller->getMaster() == this->getPlayer()->getController())
                return true;
        }
        return false;
    }

    // A function that returns a slave of the pawn's controller
    Controller* Pawn::getSlave(){
        for (FormationController* controller : ObjectList<FormationController>())
        {
            if (this->hasHumanController() && controller->getMaster() == this->getPlayer()->getController())
                return controller->getController();
        }
        return nullptr;
    }


    void Pawn::setExplosionSound(const std::string &explosionSound)
    {
        if(explosionSound_ )
            explosionSound_->setSource(explosionSound);
        else
            assert(0); // This should never happen, because soundpointer is only available on master
    }

    const std::string& Pawn::getExplosionSound()
    {
        if( explosionSound_ )
            return explosionSound_->getSource();
        else
            assert(0);
        return BLANKSTRING;
    }
}
