/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

#include "HUDWeaponSystem.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "weaponsystem/WeaponSystem.h"
#include "weaponsystem/WeaponPack.h"
#include "weaponsystem/Weapon.h"
#include "util/Convert.h"
#include "core/class/Super.h"

namespace orxonox
{
    RegisterClass(HUDWeaponSystem);

    HUDWeaponSystem::HUDWeaponSystem(Context* context) : OrxonoxOverlay(context)
    {
        RegisterObject(HUDWeaponSystem);

        weaponModeHUDSize_ = Vector2(0.0f,0.0f);
        weaponModeHUDActualSize_ = Vector2(0.0f,0.0f);        

        weapons_.clear();
        hudWeapons_.clear();
    }

    HUDWeaponSystem::~HUDWeaponSystem()
    {
        if (this->isInitialized())
        {
            destroyHUDChilds();
        }
    }

    void HUDWeaponSystem::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(HUDWeaponSystem, XMLPort, xmlelement, mode);

        XMLPortParam(HUDWeaponSystem, "weaponModeHUDSize", setWeaponModeHUDSize, getWeaponModeHUDSize, xmlelement, mode);
    }

    void HUDWeaponSystem::positionChanged()
    {
        OrxonoxOverlay::positionChanged();

        positionHUDChilds();
    }

    void HUDWeaponSystem::sizeChanged()
    {
        OrxonoxOverlay::sizeChanged();

        weaponModeHUDActualSize_ = this->getActualSize();

        positionHUDChilds();
    }

    void HUDWeaponSystem::changedOwner()
    {
        SUPER(HUDWeaponSystem, changedOwner);

        this->owner_ = orxonox_cast<Pawn*>(this->getOwner());

        updateWeaponList();
    }

    void HUDWeaponSystem::changedOverlayGroup()
    {
        SUPER(HUDWeaponSystem, changedOverlayGroup);
    }   

    void HUDWeaponSystem::changedVisibility()
    {
        SUPER(HUDWeaponSystem, changedVisibility);

        bool visible = this->isVisible();

        for (std::vector<WeakPtr<HUDWeapon> >::iterator it = hudWeapons_.begin(); it != hudWeapons_.end(); ++it)
        {
            (*it)->changedVisibility(); //inform all Child Overlays that our visibility has changed
            (*it)->setVisible(visible);
        }
    }

    void HUDWeaponSystem::changedName()
    {
        SUPER(HUDWeaponSystem, changedName);
    }  

    void HUDWeaponSystem::updateWeaponList()
    {
        if (owner_ == nullptr)
        {
            return;
        }

        weapons_.clear();

        destroyHUDChilds();

        std::vector<WeaponPack*>* weaponPacks = owner_->getAllWeaponPacks();

        for (std::vector<WeaponPack*>::const_iterator itPacks = weaponPacks->begin(); itPacks != weaponPacks->end(); ++itPacks)
        {
            std::vector<Weapon*>* weapons = (*itPacks)->getAllWeapons();

            for (std::vector<Weapon*>::const_iterator itWeapons = weapons->begin(); itWeapons != weapons->end(); ++itWeapons)
            {
                this->weapons_.push_back(*itWeapons);
            }
        }

        createHUDChilds();
        positionHUDChilds();
    }

    void HUDWeaponSystem::createHUDChilds()
    {
        int positionIndex = 0;

        for (std::vector<WeakPtr<Weapon> >::iterator it = weapons_.begin(); it != weapons_.end(); ++it)
        {
            HUDWeapon* hudWeapon = new HUDWeapon(this->getContext());
            hudWeapon->setOwner(owner_);
            hudWeapon->setOverlayGroup(this->getOverlayGroup());
            hudWeapon->setVisible(this->isVisible());
            hudWeapon->setWeapon(*it);
            hudWeapon->setAspectCorrection(false);
            hudWeapon->setPickPoint(Vector2(0.0f,0.0f));

            hudWeapons_.push_back(hudWeapon);

            ++ positionIndex;
        }
    }      

    void HUDWeaponSystem::positionHUDChilds()
    {
        int positionIndex = 0;
        Vector2 offset = this->getPosition();

        for (std::vector<WeakPtr<HUDWeapon> >::iterator it = hudWeapons_.begin(); it != hudWeapons_.end(); ++it)
        {
            (*it)->setPositionOffset(offset);
            (*it)->setWeaponIndex(positionIndex);
            (*it)->setWeaponModeHUDActualSize(this->weaponModeHUDActualSize_);

            ++ positionIndex;
        }
    }  

    void HUDWeaponSystem::destroyHUDChilds()
    {
        for (std::vector<WeakPtr<HUDWeapon> >::iterator it = hudWeapons_.begin(); it != hudWeapons_.end(); ++it)
        {
            (*it)->destroy();
        } 

        hudWeapons_.clear();     
    }
}
