/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Manuel Meier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file HoverWall.cc
    @brief Represents one Wall piece in the Hover Game
*/

#include "HoverWall.h"

#include "core/CoreIncludes.h"
#include "core/GameMode.h"
#include "graphics/Model.h"
#include "gametypes/Gametype.h"



#include "core/XMLPort.h"

namespace orxonox
{
    RegisterClass(HoverWall);

    HoverWall::HoverWall(Context* context) : StaticEntity(context)
    {
        RegisterObject(HoverWall);
        model_ = NULL;
        cs_ = NULL;
    }


    /**
    @brief
        Constructor of a HoverWall
    @param x
        x-Coordinate of the Square that the Wall is attached to, 0-9, Origin is Bottom left
    @param y
        y-Coordinate of the Square that the Wall is attached to, 0-9, Origin is Bottom left
    @param orientation
            Wall on the right side (1) or on top (2) of this square, 0-1        
    */
    HoverWall::HoverWall(Context* context, int x, int y, int cellSize, int cellHeight, int orientation) : StaticEntity(context)
    {
        RegisterObject(HoverWall);

        int xSize_, zSize_, xPos_, zPos_;

        if(orientation == 1){
            xSize_ = cellSize/2;
            zSize_ = 2;
            zPos_ = x*cellSize;
            xPos_ = y*cellSize-cellSize/2;
        }
        else{
            xSize_ = 2;
            zSize_ = cellSize/2;
            zPos_ = x*cellSize-cellSize/2;
            xPos_ = y*cellSize;
        }


        model_ = new Model(context);
        model_->setMeshSource("CuboidBody.mesh");
        model_->setScale3D(Vector3(xSize_*1.0f, cellHeight*1.0f, zSize_*1.0f));
        model_->setPosition(Vector3(xPos_*1.0f, 0.0f, zPos_*1.0f));

        this->attach(model_);

        this->enableCollisionCallback();
        this->setCollisionResponse(true);
        this->setCollisionType(Static);

        cs_ = new BoxCollisionShape(context);
        cs_->setHalfExtents(Vector3(xSize_*1.0f, cellHeight*1.0f, zSize_*1.0f));
        cs_->setPosition(Vector3(xPos_*1.0f, 0.0f, zPos_*1.0f));

        this->attachCollisionShape(cs_);
    }

    /**
    @brief
        Destructor.
    */
    HoverWall::~HoverWall()
    {

    }

    //xml port for loading height and width of the platform, unused
    void HoverWall::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(HoverWall, XMLPort, xmlelement, mode);
    }
}
