/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Manuel Meier
 *   Co-authors:
 *      ...
 *
 */

#ifndef _HoverOrigin_H__
#define _HoverOrigin_H__

#include "HoverPrereqs.h"

#include <string>

#include <util/Math.h>

#include "worldentities/StaticEntity.h"

namespace orxonox
{
    
    /**
    @brief
        The HoverOrigin implements the playing field @ref orxonox::Hover "Hover" takes place in and allows for many parameters of the minigame to be set.
        The playing field resides in the x,z-plane, with the x-axis being the horizontal axis and the z-axis being the vertical axis.
        
        Various parameters can be set:
        - The <b>dimension</b> is a vector, that defines the width and height of the playing field. The default is <em>(200, 120)</em>.
        - The <b>balltemplate</b> is a template that is applied to the @ref orxonox::HoverPlatform "HoverPlatform", it can be used to attach different things to it, e.g. its @ref orxonox::Model "Model". See below for a usage example.
        - The <b>battemplate</b> is a template that is applied to the @ref orxonox::HoverPlatform "HoverFigure", it can be used to attach different things to it, e.g. its @ref orxonox::Model "Model". See below for a usage example.
        - The <b>ballspeed</b> is the speed with which the @ref orxonox::HoverPlatform "HoverPlatform" moves. The default is <em>100</em>.
        - The <b>ballaccfactor</b> is the acceleration factor for the @ref orxonox::HoverPlatform "HoverPlatform". The default is <em>1.0</em>.
        - The <b>batspeed</b> is the speed with which the @ref orxonox::HoverFigure "HoverFigures" move. The default is <em>60</em>.
        - The <b>batlength</b> is the length of the @ref orxonox::HoverFigure "HoverFigures" as the percentage of the height of the playing field. The default is <em>0.25</em>.
        
        An example in XML of the HoverOrigin would be:
        
        First the needed templates:
        The template for the @ref orxonox::HoverPlatform "HoverPlatform".
        @code
        <Template name="Hoverball">
          <HoverPlatform>
            <attached>
              <Model mesh="sphere.mesh" scale="2" />
              <ParticleSpawner name="hiteffect" position="0,0,0" source="Orxonox/sparks2" lifetime="0.01" autostart="0" mainstate="spawn" />
            </attached>
            <eventlisteners>
              <EventTarget target="hiteffect" />
            </eventlisteners>
          </HoverPlatform>
        </Template>
        @endcode
        As can be seen, a sphere is attached as the @ref orxonox::Model "Model" for the @ref orxonox::HoverPlatform "HoverPlatform", and also an @ref orxonox::EventListener "EventListener" that triggers a @ref orxonox::ParticleSpawner "ParticleSpawner", whenever the ball hits the boundaries is attached.
        
        Additionally the template for the @ref orxonox::HoverFigure "HoverFigure".
        @code
        <Template name="Hoverbatcameras" defaults="0">
          <HoverFigure>
            <camerapositions>
              <CameraPosition position="0,200,0" pitch="-90" absolute="true" />
            </camerapositions>
          </HoverFigure>
        </Template>

        <Template name="Hoverbat">
          <HoverFigure camerapositiontemplate=Hoverbatcameras>
            <attached>
              <Model position="0,0,3" mesh="cube.mesh" scale3D="14,2,2" />
            </attached>
          </HoverFigure>
        </Template>
        @endcode
        As can be seen, there are actually two templates. The first template is needed to set the camera for the @ref orxonox::HoverFigure "HoverFigure". The second template ist the actual template for the @ref orxonox::HoverFigure "HoverFigure", the template for the camera position is added and a @ref orxonox::Model "Model" for the @ref orxonox::HoverFigure "HoverFigure" is attached.
        propellerTemplate_
        Finally the HoverOrigin is created.
        @code
        <HoverOrigin name="Hovercenter" dimension="200,120" balltemplate="Hoverball" battemplate="Hoverbat" ballspeed="200" ballaccfactor="1.0" batspeed="130" batlength="0.25">
          <attached>
            <Model position="0,0,60" mesh="cube.mesh" scale3D="105,1,1" />
            <Model position="0,0,-60" mesh="cube.mesh" scale3D="105,1,1" />
          </attached>
        </HoverOrigin>
        @endcode
        All parameters are specified. And also two @ref orxonox::Model "Models" (for the upper and lower boundary) are attached.
        
        For a more elaborate example, have a look at the <code>Hover.oxw</code> level file.

    */
    class _HoverExport HoverOrigin : public StaticEntity
    {
        public:
            HoverOrigin(Context* context); //!< Constructor. Registers and initializes the object and checks whether the gametype is actually Hover.
            virtual ~HoverOrigin() {}
            virtual void XMLPort(Element& xmlelement, XMLPort::Mode mode); //!< Method to create a HoverOrigin through XML.
            
            inline void setNumCells(int numCells)
                { this->numCells_ = numCells; }
            inline int getNumCells() const
                { return this->numCells_; }

            inline void setCellSize(int cellSize)
                { this->cellSize_ = cellSize; }
            inline int getCellSize() const
                { return this->cellSize_; }

            inline void setCellHeight(int cellHeight)
                { this->cellHeight_ = cellHeight; }
            inline int getCellHeight() const
                { return this->cellHeight_; }

        private:
            void checkGametype();
       
            int numCells_;
            int cellSize_;
            int cellHeight_;
    };
}

#endif /* _HoverOrigin_H__ */
