/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Reto Grieder
 *   Co-authors:
 *      ...
 *
 */

/**
@file
@brief Definition of the OverlayGroup class.
*/

#include "OverlayGroup.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "core/command/ConsoleCommandIncludes.h"
#include "OrxonoxOverlay.h"
#include "gametypes/Gametype.h"

namespace orxonox
{
    RegisterClass(OverlayGroup);

    SetConsoleCommand("OverlayGroup", "toggleVisibility", &OverlayGroup::toggleVisibility);
    SetConsoleCommand("OverlayGroup", "show", &OverlayGroup::show);
    SetConsoleCommand("OverlayGroup", "scaleGroup",       &OverlayGroup::scaleGroup);
    SetConsoleCommand("OverlayGroup", "scrollGroup",      &OverlayGroup::scrollGroup);

    OverlayGroup::OverlayGroup(Context* context)
        : BaseObject(context)
    {
        RegisterObject(OverlayGroup);

        this->owner_ = nullptr;

        setScale(Vector2(1.0, 1.0));
        setScroll(Vector2(0.0, 0.0));
    }

    OverlayGroup::~OverlayGroup()
    {
        for (OrxonoxOverlay* hudElement : hudElements_)
            hudElement->destroy();
        this->hudElements_.clear();
    }

    /**
    @brief
        Loads the group and all its children OrxonoxOverlays.
    @copydoc
        BaseObject::XMLPort()
    */
    void OverlayGroup::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(OverlayGroup, XMLPort, xmlelement, mode);

        XMLPortParam(OverlayGroup, "scale",  setScale,  getScale,  xmlelement, mode);
        XMLPortParam(OverlayGroup, "scroll", setScroll, getScroll, xmlelement, mode);
        // loads all the child elements
        XMLPortObject(OverlayGroup, OrxonoxOverlay, "", addElement, getElement, xmlelement, mode);
    }

    //! Scales every element in the set.
    void OverlayGroup::setScale(const Vector2& scale)
    {
        for (OrxonoxOverlay* hudElement : hudElements_)
            hudElement->scale(scale / this->scale_);
        this->scale_ = scale;
    }

    //! Scrolls every element in the set.
    void OverlayGroup::setScroll(const Vector2& scroll)
    {
        for (OrxonoxOverlay* hudElement : hudElements_)
            hudElement->scroll(scroll - this->scroll_);
        this->scroll_ = scroll;
    }

    /**
    @brief
        Adds an element to the set (used when loading with XMLPort).
    @remarks
        The names of the OrxonoxOverlays have to be unique!
    */
    void OverlayGroup::addElement(OrxonoxOverlay* element)
    {
        hudElements_.insert(element);
        element->setOverlayGroup( this );
        if (this->owner_)
            element->setOwner(this->owner_);
    }

    /**
    @brief
        Removes an element from the map.
    @param element
        A pointer to the element that is removed.
    @return
        Returns true if there was such an element to remove, false if not.
    */
    bool OverlayGroup::removeElement(OrxonoxOverlay* element)
    {
        if(this->hudElements_.erase(element) == 0)
            return false;
        return true;
    }

    //! Returns a different element as long as index < hudElements_.size().
    OrxonoxOverlay* OverlayGroup::getElement(unsigned int index)
    {
        if (index < this->hudElements_.size())
        {
            std::set<StrongPtr<OrxonoxOverlay>>::const_iterator it = hudElements_.begin();
            for (unsigned int i = 0; i != index; ++it, ++i)
                ;
            return *it;
        }
        else
            return nullptr;
    }

    //! Changes the visibility of all elements
    void OverlayGroup::changedVisibility()
    {
        SUPER( OverlayGroup, changedVisibility );

        for (OrxonoxOverlay* hudElement : hudElements_)
            hudElement->changedVisibility(); //inform all Child Overlays that our visibility has changed
    }

    void OverlayGroup::setOwner(BaseObject* owner)
    {
        this->owner_ = owner;

        for (OrxonoxOverlay* hudElement : hudElements_)
            hudElement->setOwner(owner);
    }

    //########### Console commands ############

    /**
    @brief
        Hides/shows an overlay group by its name.
    @param name
        The name of the group defined BaseObject::setName() (usually done with the "name"
        attribute in the xml file).
    */
    /*static*/ void OverlayGroup::toggleVisibility(const std::string& name)
    {
        for (OverlayGroup* group : ObjectList<OverlayGroup>())
        {
            if (group->getName() == name)
                group->setVisible(!(group->isVisible()));
        }
    }
    
    /**
    @brief
        Shows an overlay group by its name.
    @param name
        The name of the group defined BaseObject::setName() (usually done with the "name" attribute in the xml file).
    */
    /*static*/ void OverlayGroup::show(const std::string& name)
    {
        for (OverlayGroup* group : ObjectList<OverlayGroup>())
        {
            if (group->getName() == name)
            {
                if(group->isVisible())
                    group->changedVisibility();
                else
                    group->setVisible(!(group->isVisible()));
            }
        }
    }

    /**
    @brief
        Scales an overlay group by its name.
    @param name
        The name of the group defined BaseObject::setName() (usually done with the "name"
        attribute in the xml file).
    @param scale
        The scaling factor
    */
    /*static*/ void OverlayGroup::scaleGroup(const std::string& name, float scale)
    {
        for (OverlayGroup* group : ObjectList<OverlayGroup>())
        {
            if (group->getName() == name)
                group->scale(Vector2(scale, scale));
        }
    }

    /**
    @brief
        Scrolls an overlay group by its name.
    @param name
        The name of the group defined BaseObject::setName() (usually done with the "name"
        attribute in the xml file).
    @param scroll
        The relative translation of the overlay group
    */
    /*static*/ void OverlayGroup::scrollGroup(const std::string& name, const Vector2& scroll)
    {
        for (OverlayGroup* group : ObjectList<OverlayGroup>())
        {
            if (group->getName() == name)
                group->scroll(scroll);
        }
    }
}
