/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file SplitGun.cc
    @brief Implementation of the SplitGun class.
*/

#include "SplitGun.h"

#include "core/CoreIncludes.h"
#include "core/XMLPort.h"
#include "weaponsystem/Weapon.h"
#include "weaponsystem/WeaponPack.h"
#include "weaponsystem/WeaponSystem.h"
#include "worldentities/pawns/Pawn.h"

#include "weapons/projectiles/SplitGunProjectile.h"

namespace orxonox
{
    RegisterClass(SplitGun);

    SplitGun::SplitGun(Context* context) : WeaponMode(context)
    {
        RegisterObject(SplitGun);

        this->reloadTime_ = 1.0f;
        this->damage_ = 0.0f;
        this->speed_ = 750.0f;
        this->numberOfSplits_ = 2;
        this->numberOfChilds_ = 3;
        this->splitTime_ = 0.3f;
        this->spread_ = 0.1f;
        this->damageReduction_ = 1.0f;

        this->setMunitionName("SplitMunition");
        this->setDefaultSound("sounds/Weapon_LightningGun.ogg");

        hudImageString_ = "Orxonox/WSHUD_WM_SplitGun";
    }

    SplitGun::~SplitGun()
    {
    }

    /**
    @brief
        XMLPort for the SplitGun. You can define how often the projectiles split, how many childs should be created per split, the spread and the time between two splits.
    */
    void SplitGun::XMLPort(Element& xmlelement, XMLPort::Mode mode)
    {
        SUPER(SplitGun, XMLPort, xmlelement, mode);

        XMLPortParam(SplitGun, "numberofsplits", setNumberOfSplits, getNumberOfSplits, xmlelement, mode);
        XMLPortParam(SplitGun, "numberofchilds", setNumberOfChilds, getNumberOfChilds, xmlelement, mode);
        XMLPortParam(SplitGun, "splittime", setSplitTime, getSplitTime, xmlelement, mode);
        XMLPortParam(SplitGun, "spread", setSpread, getSpread, xmlelement, mode);
        XMLPortParam(SplitGun, "damagereduction", setDamageReduction, getDamageReduction, xmlelement, mode);
    }

    /**
    @brief
        Fires the weapon. Creates a projectile and fires it.
    */
    void SplitGun::fire()
    {
        SplitGunProjectile* projectile = new SplitGunProjectile(this->getContext());
        projectile->setMaterial("Flares/energyflare");

        this->computeMuzzleParameters(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn()->getAimPosition());
        projectile->setOrientation(this->getMuzzleOrientation());
        projectile->setPosition(this->getMuzzlePosition());
        projectile->setVelocity(this->getMuzzleDirection() * this->speed_);

        // Pass important information to the projectile: Number of splits, Number of childs, split time, spread and the damage reduction
        projectile->setNumberOfSplits(getNumberOfSplits());
        projectile->setNumberOfChilds(getNumberOfChilds());
        projectile->setSplitTime(getSplitTime());
        projectile->setSpread(getSpread());
        projectile->setDamageReduction(getDamageReduction());

        projectile->setShooter(this->getWeapon()->getWeaponPack()->getWeaponSystem()->getPawn());
        projectile->setDamage(this->getDamage());
        projectile->setShieldDamage(this->getShieldDamage());
        projectile->setHealthDamage(this->getHealthDamage());
    }
}
