/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabian 'x3n' Landau
 *   Co-authors:
 *      ...
 *
 */

#include "PluginManager.h"

#include <fstream>

#include "SpecialConfig.h"
#include "Plugin.h"
#include "PluginReference.h"
#include "core/ApplicationPaths.h"
#include "core/command/ConsoleCommandIncludes.h"
#include "core/object/Context.h"

namespace orxonox
{
    static const std::string __CC_PluginManager_load_name  = "load";
    static const std::string __CC_PluginManager_unload_name  = "unload";

    SetConsoleCommand("PluginManager", __CC_PluginManager_load_name, &PluginManager::loadPlugin);
    SetConsoleCommand("PluginManager", __CC_PluginManager_unload_name, &PluginManager::unloadPlugin);

    PluginManager* PluginManager::singletonPtr_s  = nullptr;

    PluginManager::PluginManager()
    {
        ModifyConsoleCommand("PluginManager", __CC_PluginManager_load_name).setObject(this);
        ModifyConsoleCommand("PluginManager", __CC_PluginManager_unload_name).setObject(this);
    }

    PluginManager::~PluginManager()
    {
        ModifyConsoleCommand("PluginManager", __CC_PluginManager_load_name).setObject(nullptr);
        ModifyConsoleCommand("PluginManager", __CC_PluginManager_unload_name).setObject(nullptr);

        for (std::map<std::string, PluginReference*>::iterator it = this->references_.begin(); it != this->references_.end(); ++it)
            delete it->second;
        for (std::map<std::string, Plugin*>::iterator it = this->plugins_.begin(); it != this->plugins_.end(); ++it)
            delete it->second;
    }

    void PluginManager::findPlugins()
    {
        const std::vector<std::string>& pluginPaths = ApplicationPaths::getInstance().getPluginPaths();
        for (std::vector<std::string>::const_iterator it = pluginPaths.begin(); it != pluginPaths.end(); ++it)
        {
            std::string name;
            std::string libraryName = (*it);
            std::string filename = libraryName +  + specialConfig::pluginExtension;
            std::ifstream infile(filename.c_str());
            if (infile >> name)
            {
                orxout(internal_info) << "Found plugin with name '" << name << "' in module " << libraryName << endl;
                this->plugins_[name] = new Plugin(name, libraryName);
            }
            else
            {
                orxout(internal_warning) << "Could not read plugin file " << filename << endl;
            }
        }
    }

    void PluginManager::referencePlugin(const std::string& name)
    {
        Plugin* plugin = this->plugins_[name];
        if (plugin != nullptr)
            plugin->load();
        else
            orxout(internal_warning) << "Cannot find plugin with name " << name << endl;
    }

    void PluginManager::dereferencePlugin(const std::string& name)
    {
        Plugin* plugin = this->plugins_[name];
        if (plugin != nullptr)
            plugin->unload();
        else
            orxout(internal_warning) << "Cannot find plugin with name " << name << endl;
    }

    /**
     * @brief Console command to manually load a plugin. The plugin stays loaded until @ref unloadPlugin is called.
     */
    void PluginManager::loadPlugin(const std::string& name)
    {
        if (this->references_[name] == nullptr)
        {
            this->references_[name] = new PluginReference(name);
        }
        else
            orxout(internal_warning) << "Plugin " << name << " is already loaded" << endl;
    }

    /**
     * @brief Console command to unload a plugin if it was previously loaded manually by calling @ref loadPlugin.
     * Does not unload the plugin immediately if it is still used by another @ref PluginReference (e.g. by a @ref Level).
     */
    void PluginManager::unloadPlugin(const std::string& name)
    {
        PluginReference* reference = this->references_[name];
        if (reference != nullptr)
        {
            this->references_[name] = nullptr;
            delete reference;
        }
        else
            orxout(internal_warning) << "Plugin " << name << " is already unloaded" << endl;
    }
}
