/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file MunitionPickup.cc
    @brief Implementation of the MunitionPickup class.
*/

#include "MunitionPickup.h"

#include <sstream>
#include "core/CoreIncludes.h"
#include "core/XMLPort.h"

#include "worldentities/pawns/Pawn.h"

namespace orxonox
{
    RegisterClass(MunitionPickup);

    /**
    @brief
        Constructor. Registers the object and initializes the member variables.
    */
    MunitionPickup::MunitionPickup(Context* context) : Pickup(context)
    {
        RegisterObject(MunitionPickup);

        this->initialize();
    }

    /**
    @brief
        Destructor.
    */
    MunitionPickup::~MunitionPickup()
    {
    }

    /**
    @brief
        Initializes the member variables.
    */
    void MunitionPickup::initialize(void)
    {
        //Defines who is allowed to pick up the pickup.
        this->addTarget(ClassIdentifier<Pawn>::getIdentifier());
    }

    /**
    @brief
        Method for creating a MunitionPickup object through XML.
    */
    void MunitionPickup::XMLPort(Element& xmlelement, orxonox::XMLPort::Mode mode)
    {
        SUPER(MunitionPickup, XMLPort, xmlelement, mode);

        XMLPortObject(MunitionPickup, MunitionContainer, "munitioncontainers", addMunitionContainer, getMunitionContainer, xmlelement, mode);
    }

    void MunitionPickup::addMunitionContainer(MunitionContainer* munitionContainer)
    {
        OrxAssert(munitionContainer != NULL, "The munitionContainer cannot be NULL.");
        this->munitionContainers_.push_back(munitionContainer);
    }

    MunitionContainer* MunitionPickup::getMunitionContainer(unsigned int index)
    {
        if(this->munitionContainers_.size() >= index)
            return NULL;
        else
            return this->munitionContainers_[index];
    }

    /**
    @brief
        Is called when the pickup has transisted from used to unused or the other way around.
    */
    void MunitionPickup::changedUsed(void)
    {
        SUPER(MunitionPickup, changedUsed);

        Pawn* pawn = this->carrierToPawnHelper();

        if(pawn == NULL) // If the PickupCarrier is no Pawn, then this pickup is useless and therefore is destroyed.
            this->Pickupable::destroy();

        // If the pickup has transited to used.
        if(this->isUsed())
        {
            for(std::vector<MunitionContainer*>::iterator it = this->munitionContainers_.begin(); it != this->munitionContainers_.end(); ++it)
            {
                //Get pointer to the appropriate munition
                SubclassIdentifier<Munition> identifier = (*it)->getMunitionType();
                Munition* munition = pawn->getMunition(&identifier);
                if (munition)
                {
                    // Add munition and magzines
                    munition->addMunition((*it)->getMunitionAmount());
                    munition->addMagazines((*it)->getMagazinesAmount());
                }
                (*it)->destroy();
            }
            // This will destroy the pickp
            this->setUsed(false);
        }
        else
        {
            this->Pickupable::destroy();
        }
    }    

    /**
    @brief
        Helper to transform the PickupCarrier to a Pawn, and throw an error message if the conversion fails.
    @return
        A pointer to the Pawn, or NULL if the conversion failed.
    */
    Pawn* MunitionPickup::carrierToPawnHelper(void)
    {
        PickupCarrier* carrier = this->getCarrier();
        Pawn* pawn = orxonox_cast<Pawn*>(carrier);

        if(pawn == NULL)
        {
            orxout(internal_error, context::pickups) << "Invalid PickupCarrier in MunitionPickup." << endl;
        }
        return pawn;
    }      
}

