/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer

   This code was inspired by the command_node.cc code from Christian Meyer in revision
   4386 and earlier.
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_EVENT

#include "key_mapper.h"

#include "event_def.h"

#include "orxonox_globals.h"
#include "parser/ini_parser/ini_parser.h"
#include "parser/preferences/preferences.h"
#include "key_names.h"
#include "event_def.h"
#include "debug.h"



ObjectListDefinition(KeyMapper);

/* initialize all variables to a reasonable value*/
int KeyMapper::PEV_FORWARD           = EV_UNKNOWN;
int KeyMapper::PEV_BACKWARD          = EV_UNKNOWN;
int KeyMapper::PEV_LEFT              = EV_UNKNOWN;
int KeyMapper::PEV_RIGHT             = EV_UNKNOWN;
int KeyMapper::PEV_UP                = EV_UNKNOWN;
int KeyMapper::PEV_DOWN              = EV_UNKNOWN;
int KeyMapper::PEV_ROLL_LEFT         = EV_UNKNOWN;
int KeyMapper::PEV_ROLL_RIGHT        = EV_UNKNOWN;
int KeyMapper::PEV_STRAFE_LEFT       = EV_UNKNOWN;
int KeyMapper::PEV_STRAFE_RIGHT      = EV_UNKNOWN;
int KeyMapper::PEV_JUMP              = EV_UNKNOWN;
int KeyMapper::PEV_ACTION            = EV_UNKNOWN;
int KeyMapper::PEV_CROUCH            = EV_UNKNOWN;

int KeyMapper::PEV_FIRE1             = EV_UNKNOWN;
int KeyMapper::PEV_FIRE2             = EV_UNKNOWN;
int KeyMapper::PEV_PREVIOUS_WEAPON   = EV_UNKNOWN;
int KeyMapper::PEV_NEXT_WEAPON       = EV_UNKNOWN;

int KeyMapper::PEV_CHANGE_SHIP       = EV_UNKNOWN;

int KeyMapper::PEV_VIEW0             = EV_UNKNOWN;
int KeyMapper::PEV_VIEW1             = EV_UNKNOWN;
int KeyMapper::PEV_VIEW2             = EV_UNKNOWN;
int KeyMapper::PEV_VIEW3             = EV_UNKNOWN;
int KeyMapper::PEV_VIEW4             = EV_UNKNOWN;
int KeyMapper::PEV_VIEW5             = EV_UNKNOWN;

int KeyMapper::PEV_NEXT_WORLD        = EV_UNKNOWN;
int KeyMapper::PEV_PREVIOUS_WORLD    = EV_UNKNOWN;

int KeyMapper::PEV_PAUSE             = EV_UNKNOWN;
int KeyMapper::PEV_QUIT              = EV_UNKNOWN;



//! this is the mapping array from names to ids: enter all orxonox.conf keys here
/** @todo use globals.h for this.... everything is done there for those Options,
 * and you do not have to care about The namings, as they might change
 */
KeyMapper::KeyMapping KeyMapper::map[] = {
      {&KeyMapper::PEV_FORWARD,              CONFIG_NAME_PLAYER_FORWARD,         SDLK_w},
      {&KeyMapper::PEV_BACKWARD,             CONFIG_NAME_PLAYER_BACKWARD,        SDLK_s},
      {&KeyMapper::PEV_UP,                   CONFIG_NAME_PLAYER_UP,              SDLK_r},
      {&KeyMapper::PEV_DOWN,                 CONFIG_NAME_PLAYER_DOWN,            SDLK_f},
      {&KeyMapper::PEV_LEFT,                 CONFIG_NAME_PLAYER_LEFT,            SDLK_a},
      {&KeyMapper::PEV_RIGHT,                CONFIG_NAME_PLAYER_RIGHT,           SDLK_d},
      {&KeyMapper::PEV_ROLL_RIGHT,           CONFIG_NAME_PLAYER_ROLL_LEFT,       SDLK_z},
      {&KeyMapper::PEV_ROLL_LEFT,            CONFIG_NAME_PLAYER_ROLL_RIGHT,      SDLK_c},
      {&KeyMapper::PEV_STRAFE_LEFT,          CONFIG_NAME_PLAYER_STRAFE_LEFT,     SDLK_q},
      {&KeyMapper::PEV_STRAFE_RIGHT,         CONFIG_NAME_PLAYER_STRAFE_RIGHT,    SDLK_e},
      {&KeyMapper::PEV_JUMP,                 CONFIG_NAME_PLAYER_JUMP,            SDLK_SPACE},
      {&KeyMapper::PEV_ACTION,               CONFIG_NAME_PLAYER_ACTION,          SDLK_u},
      {&KeyMapper::PEV_CROUCH,               CONFIG_NAME_PLAYER_CROUCH,          SDLK_LCTRL},


      {&KeyMapper::PEV_FIRE1,                CONFIG_NAME_PLAYER_FIRE,            EV_MOUSE_BUTTON_LEFT},
      {&KeyMapper::PEV_FIRE2,                CONFIG_NAME_PLAYER_FIRE2,           EV_MOUSE_BUTTON_RIGHT},
      {&KeyMapper::PEV_NEXT_WEAPON,          CONFIG_NAME_PLAYER_NEXT_WEAPON,     EV_MOUSE_BUTTON_WHEELUP},
      {&KeyMapper::PEV_PREVIOUS_WEAPON,      CONFIG_NAME_PLAYER_PREV_WEAPON,     EV_MOUSE_BUTTON_WHEELDOWN},

      {&KeyMapper::PEV_CHANGE_SHIP,          CONFIG_NAME_PLAYER_CHANGE_SHIP,     SDLK_g},


      {&KeyMapper::PEV_VIEW0,                CONFIG_NAME_VIEW0,                  SDLK_1},
      {&KeyMapper::PEV_VIEW1,                CONFIG_NAME_VIEW1,                  SDLK_2},
      {&KeyMapper::PEV_VIEW2,                CONFIG_NAME_VIEW2,                  SDLK_3},
      {&KeyMapper::PEV_VIEW3,                CONFIG_NAME_VIEW3,                  SDLK_4},
      {&KeyMapper::PEV_VIEW4,                CONFIG_NAME_VIEW4,                  SDLK_5},
      {&KeyMapper::PEV_VIEW5,                CONFIG_NAME_VIEW5,                  SDLK_6},

      {&KeyMapper::PEV_NEXT_WORLD,           CONFIG_NAME_NEXT_WORLD,             SDLK_x},
      {&KeyMapper::PEV_PREVIOUS_WORLD,       CONFIG_NAME_PREV_WORLD,             SDLK_z},

      {&KeyMapper::PEV_PAUSE,                CONFIG_NAME_PAUSE,                  SDLK_p},
      {&KeyMapper::PEV_QUIT,                 CONFIG_NAME_QUIT,                   SDLK_ESCAPE},
      {NULL, "", 0}
    };



/**
 *  standard constructor
*/
KeyMapper::KeyMapper ()
{
  this->registerObject(this, KeyMapper::_objectList);
}


/**
 *  standard deconstructor
*/
KeyMapper::~KeyMapper ()
{}


/**
 *  loads new key bindings from a file
 * @param filename: The path and name of the file to load the bindings from
*/
void KeyMapper::loadKeyBindings (const std::string& fileName)
{
  IniParser parser(fileName);
  this->loadKeyBindings(&parser);
}

void KeyMapper::loadKeyBindings(IniParser* iniParser)
{
  IniParser::Section* section = iniParser->getSection (CONFIG_SECTION_CONTROL);
  if( section == NULL)
  {
    PRINTF(1)("Could not find key bindings " CONFIG_SECTION_CONTROL "\n");
    return;
  }
  int* index;


  for(IniParser::Entry::iterator entry = section->begin();
      entry != section->end();
      ++entry)
  {
    PRINTF(3)("Keys: Parsing %s, %s now.\n", (*entry).name().c_str(), (*entry).value().c_str());
    // map the name to an sdl index
    index = nameToIndex ((*entry).value());
    // map the index to a internal name
    this->mapKeys((*entry).name(), index);
  }


  // PARSE MISC SECTION
  //   if( !iniParser->getSection (CONFIG_SECTION_MISC_KEYS))
  //   {
  //     PRINTF(1)("Could not find key bindings" CONFIG_SECTION_MISC_KEYS "\n");
  //     return;
  //   }
  //
  //   iniParser->firstVar();
  //   while( iniParser->getCurrentName() != "" )
  //   {
  //     PRINTF(3)("MISC: Parsing %s, %s now.\n", iniParser->getCurrentName(), iniParser->getCurrentValue());
  //     index = nameToIndex (iniParser->getCurrentValue());
  //     this->mapKeys(iniParser->getCurrentName(), index);
  //     iniParser->nextVar();
  //   }
}

void KeyMapper::loadKeyBindings()
{
  if( !Preferences::getInstance()->sectionExists(CONFIG_SECTION_CONTROL))
  {
    PRINTF(1)("Could not find key bindings " CONFIG_SECTION_CONTROL "\n");
    return;
  }
  int* index;

  std::list<std::string> keys = Preferences::getInstance()->listKeys(CONFIG_SECTION_CONTROL);
  for ( std::list<std::string>::const_iterator it = keys.begin(); it!=keys.end(); it++ )
  {
    PRINTF(3)("Keys: Parsing %s, %s now.\n", it->c_str(), Preferences::getInstance()->getString(CONFIG_SECTION_CONTROL, *it, "").c_str());
    // map the name to an sdl index
    index = nameToIndex (Preferences::getInstance()->getString(CONFIG_SECTION_CONTROL, *it, ""));
    // map the index to a internal name
    this->mapKeys(*it, index);
  }
}

/**
 *  this function looks up name to key index
 * @param the name of the button
*/
int* KeyMapper::nameToIndex (const std::string& name)
{
  coord[0] = -1;
  coord[1] = -1;
  int c;
  if( (c = keynameToSDLK (name)) != -1)
  {
    coord[1] = c;
    coord[0] = 0;
  }
  if( (c = buttonnameToSDLB (name)) != -1)
  {
    coord[1] = c;
    coord[0] = 1;
  }
  return coord;
}


/**
 *  the function maps name to key ids
 * @param name of the key
 * @param id of the key
*/
void KeyMapper::mapKeys(const std::string& name, int* index)
{
  for( int i = 0; map[i].pValue != NULL; ++i )
  {
    if( name == map[i].pName)
    {
      if( index[0] == 0)
      {
        *map[i].pValue = index[1];
        PRINTF(4)("Mapping %s to '%s' (id %i)\n", name.c_str(), SDLKToKeyname(index[1]).c_str(), index[1]);
        break;
      }
      else
      {
        *map[i].pValue = index[1];
        PRINTF(4)("Mapping %s to '%s' (id %i)\n", name.c_str(), SDLBToButtonname(index[1]).c_str(), index[1]);
        break;
      }
    }
  }
}


/**
 *  this function gives some debug information about the key mapper class
*/
void KeyMapper::debug()
{
  PRINT(0)("\n==========================| KeyMapper::debug() |===\n");
  for(int i = 0; map[i].pValue != NULL; ++i)
  {
    PRINT(0)("%s = %i\n",map[i].pName.c_str(), *map[i].pValue);
  }
  PRINT(0)("=======================================================\n");
}
