/*!
 * @file weapon_manager.h
 * every big WorldEntity has the ability to carry many different Weapons.
 * for this to be easy there is the WeaponManager, that handels these weapons,
 * and changes between them.
 *
 *
 *
 * @TODO 1. WeaponManager should also handle a List of availiableWeapons.
 */

#ifndef _WEAPON_MANAGER_H
#define _WEAPON_MANAGER_H

#include "base_object.h"

#include "crosshair.h"
#include "weapon.h"

#include "count_pointer.h"
#include "ammo_container.h"


// FORWARD DECLARATION
template <class T> class tAnimation;
class WeaponSlot;

#define    WM_MAX_SLOTS            10             //!< How many slots the WeaponManager has at its max
#define    WM_MAX_CONFIGS          4              //!< The maximum number of predefined Configurations
#define    WM_MAX_LOADED_WEAPONS   20             //!< The

//! This is a special class, that can handle many different Weapons of a ship/man/whatever.
/**
 * this class is designed to interactively changeing multiple weapons (or just one),
 * and to allow the Weapon itself to enable/disable itself.
 *
 * How to configure
 * 1. set the default values.
 * 2. define weapons. connect them to the WeaponManager's configurations (have a look at "player.cc", to see how it works)
 * 3. go on and run :)....
 */
class WeaponManager : public BaseObject {
  ObjectListDeclaration(WeaponManager);

  public:
    WeaponManager(WorldEntity* parent);
    WeaponManager(const TiXmlElement* root);
    virtual ~WeaponManager();

    void init();
    virtual void loadParams(const TiXmlElement* root);
    void loadWeapons(const TiXmlElement* root);

    void showCrosshair();
    void hideCrosshair();
    void setRotationSpeed(float speed);

    void setSlotCount(unsigned int slotCount);
    unsigned int getSlotCount() const { return this->slotCount; };
    // setting up the WeaponManager with the following functions
    void setSlotPosition(int slot, const Vector& position, PNode* parent = NULL);
//     inline void setSlotPosition(float slot, float x, float y,float z) {setSlotPosition((int)slot, Vector(x,y,z));};
    void setSlotDirection(int slot, const Quaternion& rotation);
    /** @param slot the slot to get the relative position from @returns the relative position of the Carrier to the Slot */
    const Vector& getSlotPosition(int slot) const;
    void setSlotCapability(int slot, unsigned long slotCapability);
    /** @param slot the slot to get the capabilities from @returns the capabilies */
    long getSlotCapability(int slot) const;

    void setParentEntity(WorldEntity* parent);
    WorldEntity* getParentEntity() const { return this->parentEntity; };

    void setParentNode(PNode* node);
    /** @returns the Parent (carrier) of this WeaponManager */
    PNode* getParentNode() const { return this->parentNode; };

    bool addWeapon(Weapon* weapon, int configID = -1, int slotID = -1);
    void removeWeapon(Weapon* weapon, int configID = -1);

    inline void createWeaponSlot(int slot, float x, float y, float z, long capability) {
        this->setSlotPosition(slot, Vector( x, y, z));
        this->setSlotCapability(slot, capability); };

    inline void addWeaponToSlot(int config, int slot, const std::string& weaponName){
      this->addWeapon( Weapon::createWeapon( weaponName ), config, slot); };


    Weapon* getWeapon(int slotID) const;

    // FIXME ::
//    bool hasFreeSlot(int configID, long capability = WTYPE_ALL) { return ( getNextFreeSlot(configID, capability ) != -1)? true : false; };

    void nextWeaponConfig();
    void previousWeaponConfig();
    void changeWeaponConfig(int weaponConfig);

    float increaseAmmunition(const ClassID& projectileType, float ammo);
    float increaseAmmunition(const Weapon* weapon, float ammo);

    /** @returns a fixed target namely the Crosshair's 3D position */
    inline PNode* getFixedTarget() const { return this->crosshair; };

    void fire();
    //! @TODO: implement this function (maybe also in Weapon itself)
    void releaseFire();
    //inline void setFire() { this->bFire = true; };

    void tick(float dt);
    void draw() const;

    void debug() const;

 // private:
    int getNextFreeSlot(int configID, long capability = WTYPE_ALL);
    CountPointer<AmmoContainer>& getAmmoContainer(const ClassID& projectileType);
    CountPointer<AmmoContainer>& getAmmoContainer(const Weapon* weapon);

  private:
    WorldEntity*            parentEntity;                             //!< The parent, this WeaponManager is connected to.
    PNode*                  parentNode;                               //!< The parented Node the WeaponManager is connected to. (by default == parentEntity).

    int                     slotCount;                                //!< number of weapon slots the ship has.
    int                     currentConfigID;                          //!< the currently selected config.
//     Weapon*                 configs[WM_MAX_CONFIGS][WM_MAX_SLOTS];    //!< An array of predefined configurations and assigned weapon.
//     WeaponSlot*             currentSlotConfig[WM_MAX_SLOTS];          //!< The currentConfigureation.
    WeaponSlot*             slotConfigs[WM_MAX_SLOTS];
    Weapon*                 availiableWeapons[WM_MAX_LOADED_WEAPONS]; //!< The availiable Weapons of this WeaponSlot

    bool                    weaponChange;

    Crosshair*              crosshair;                                //!< an aim.
    tAnimation<Crosshair>*  crossHairSizeAnim;                        //!< An animation for the crosshair (scaling)

    std::vector<CountPointer<AmmoContainer> > ammo;                   //!< Containers

    bool                    bFire;
};


#endif /* _WEAPON_MANAGER_H */
