/*!
 * @file world_entity.h
 * Definition of the basic WorldEntity
 */

#ifndef _WORLD_ENTITY_H
#define _WORLD_ENTITY_H

#include "p_node.h"
#include "synchronizeable.h"
#include "model.h"

#include "cr_engine.h"
#include "collision_filter.h"
#include "object_manager.h"
#include "glincl.h"

#include "aabb_tree_node.h"

#include "physics_interface.h"

#include <vector>


// FORWARD DECLARATION
namespace OrxSound { class SoundBuffer; class SoundSource; }
namespace OrxGui { class GLGuiWidget; class GLGuiBar; class GLGuiEnergyWidget; class GLGuiEnergyWidgetVertical; };
namespace CoRe { class Collision; }

class BVTree;
class BoundingVolume;
class AABBTreeNode;
class Model;
class Track;
class TiXmlElement;

class ObjectInformationFile;
class MountPoint;


//! Basis-class all interactive stuff in the world is derived from
class WorldEntity : public PNode
{
  ObjectListDeclaration(WorldEntity);

public:
  WorldEntity();
  virtual ~WorldEntity ();

  virtual void loadParams(const TiXmlElement* root);

  void loadModel(const std::string& fileName, float scaling = 1.0f, unsigned int modelNumber = 0, unsigned int obbTreeDepth = 4);
  void loadModel2(const std::string& fileN, float scal = 1.0f){this->loadModel(fileN,scal,0,4);}
  void setModel(Model* model, unsigned int modelNumber = 0);
  Model* getModel(unsigned int modelNumber = 0) const { return (this->models.size() > modelNumber)? this->models[modelNumber] : NULL; };

  void loadMountPoints(const std::string& fileName);
  inline void loadMD2Texture(const std::string& fileName) { this->md2TextureFileName = fileName; }

  void addMountPoint(MountPoint* mountPoint);
  void addMountPoint(int slot, MountPoint* mountPoint);
  void mount(int slot, WorldEntity* entity);
  void unmount(int slot);



  /** @param visibility if the Entity should be visible (been draw) */
  void setVisibility (bool visibility) { this->bVisible = visibility; };
  /** @returns true if the entity is visible, false otherwise */
  inline bool isVisible() const { return this->bVisible; };

  virtual void reset();

  virtual void postSpawn ();
  virtual void leaveWorld ();

  virtual void tick (float time);
  virtual void draw () const;
  void draw(const Model* model) const;
  void debugDrawMountPoints() const;

  /* --- Collision Detection Block  --- */
  bool buildObbTree(int depth);
  virtual void collidesWith (WorldEntity* entity, const Vector& location);
  virtual void collidesWithGround(const Vector& location);
  virtual void collidesWithGround(const Vector& feet, const Vector& ray_1, const Vector& ray_2);

  /** @returns a reference to the obb tree of this worldentity */
  inline BVTree* getOBBTree() const { return this->obbTree; };
  inline void setOBBTree(OBBTree* tree) { /*if( this->obbTree != NULL) delete this->obbTree;*/ this->obbTree = (BVTree*)tree; }
  void drawBVTree(int depth, int drawMode) const;
  inline AABB* getModelAABB() const { return (this->aabbNode)?this->aabbNode->getAABB():NULL;}

  virtual void hit(float damage, WorldEntity* killer);


  /* --- Collision Reaction Block --- */
  void subscribeReaction(CoRe::CREngine::ReactionType type, const ClassID& target1);
  void subscribeReaction(CoRe::CREngine::ReactionType type, const ClassID& target1, const ClassID& target2);
  void subscribeReaction(CoRe::CREngine::ReactionType type, const ClassID& target1, const ClassID& target2, const ClassID& target3);

  void unsubscribeReaction(CoRe::CREngine::ReactionType type);
  void unsubscribeReactions();

  /** @return true if there is at least on collision reaction subscribed */
  inline bool isReactive() const { return this->_collisionFilter.isReactive(); }

  /** @param worldEntity the world entity to be checked @returns true if there is a collisionreaction registered for the worldEntity */
  inline bool isReactive( const WorldEntity& worldEntity) const { return this->_collisionFilter(worldEntity); }
  /** @param worldEntity the world entity to be checked @param type special reaction type @returns true if collision reaction reg. */
  inline bool isReactive( const WorldEntity& worldEntity, const CoRe::CREngine::ReactionType& type) const
  { return this->_collisionFilter(worldEntity, type); }


  const CoRe::CollisionFilter& getCollisionFilter(CoRe::CREngine::ReactionType type) const { return this->_collisionFilter; }

  /** @returns true if this entity is standing on ground (BSP model) */
  bool isOnGround() const { return this->_bOnGround; }
  /** @param flag: marks if this entity is standing on ground */
  void setOnGround(bool flag) { this->_bOnGround = flag; }

  virtual void destroy( WorldEntity* killer );


  /* @returns the Count of Faces on this WorldEntity */
  //unsigned int getFaceCount () const { return (this->model != NULL)?this->model->getFaceCount():0; };
  //  void addAbility(Ability* ability);
  //  void removeAbility(Ability* ability);
  //  void setCharacterAttributes(CharacterAttributes* charAttr);
  //  CharacterAttributes* getCharacterAttributes();

  /* --- Object Manager Block --- */
  void toList(OM_LIST list);
  void toListS(const std::string& listName);

  void toReflectionList();
  void removeFromReflectionList();

  /** @returns a Reference to the objectListNumber to set. */
  OM_LIST& getOMListNumber() { return this->objectListNumber; }
  /** @returns a Reference to the Iterator */
  ObjectManager::EntityList::iterator& getEntityIterator() { return this->objectListIterator; }

  void hide() { if( this->objectListNumber != OM_DEAD) this->lastObjectListNumber = this->objectListNumber; this->toList(OM_DEAD); }
  void unhide() { if( this->objectListNumber != this->lastObjectListNumber) this->toList(this->lastObjectListNumber); }


  /* --- Character Attribute Block --- */
  /** @returns the scaling of the model */
  float getScaling(){return this->scaling;}
  /** @returns the damage dealt by this world entity */
  float getDamage() const { return this->damage; }
  /** sets the damage dealt to @param damage damage per second */
  void setDamage(float damage) { this->damage = damage; }
  /** @returns the Energy of the entity */
  float getHealth() const { return this->health; };
  /** @returns the Maximum energy this entity can be charged with */
  float getHealthMax() const { return this->healthMax; }
  float increaseHealth(float health);
  float decreaseHealth(float health);
  void increaseHealthMax(float increaseHealth);
  OrxGui::GLGuiWidget* getHealthWidget();
  OrxGui::GLGuiWidget* getShieldWidget();
  OrxGui::GLGuiWidget* getElectronicWidget();
  bool hasHealthWidget() const { return this->healthWidget != NULL; };

  float getShield() const { return this->shield; };
  float getShieldMax() const { return this->shieldMax; };
  float increaseShield(float shield);
  float decreaseShield(float shield);
  bool getShieldActive() { return this->bShieldActive; };

  float getElectronic() const { return this->electronic; };
  float getElectronicMax() const { return this->electronicMax; };
  float increaseElectronic(float electronic);
  float decreaseElectronic(float electronic);

  bool systemFailure() {  return (this->electronic < float(rand())/float(RAND_MAX) * this->electronicTH); };

  void resetElectronic() { this->electronic = this->electronicMax; };
  void resetShield()  { this->shield = this->shieldMax; this->bShieldActive = true; };
  void resetHealth() { this->health = this->healthMax; };

  void createHealthWidget();
  void createShieldWidget();
  void createElectronicWidget();

  void regen(float time);

  void loadShield(float cur, float max, float th, float regen)
      { this->setShield(cur); this->setShieldMax(max); this->setShieldTH(th); this->setShieldRegen(regen); };
  void loadHealth(float cur, float max, float regen = 0) { this->setHealth(cur); this->setHealthMax(max); this->setHealthRegen(regen); };
  void loadElectronic(float cur, float max, float th, float regen)
      { this->setElectronic(cur); this->setElectronicMax(max); this->setElectronicTH(th); this->setElectronicRegen(regen); };


  virtual void varChangeHandler( std::list<int> & id );


  //FIXME: the PhysicsInterface and the whole PEngine should probably be reviewed. Here its just used to store the vars
  /* --- Physics Interface --- */
  inline PhysicsInterface getPhysicsInterface() const { return this->physicsInterface; }
  inline float getMass() const { return this->physicsInterface.getMass(); }
  inline float getTotalMass() const { return this->physicsInterface.getTotalMass(); }
  inline void setVelocity(const Vector& vel) { this->velocity = vel; }


  /* --- Misc Stuff Block --- */
  void debugWE() { this->debugEntity(); }
  ;  ///FIXME
  void debugEntity() const;

  void pauseTrack(bool stop);

//   void updateHealthWidget();
//   void updateElectronicWidget();
//   void updateShieldWidget();

protected:
  inline void setHealth(float health) { this->health = health; this->updateHealthWidget();};
  void setHealthWidgetVisibility(bool visibility);
  void setHealthMax(float healthMax);
//   void createHealthWidget();
//   void createShieldWidget();
//   void createElectronicWidget();
  void setHealthRegen(float regen) { this->healthRegen = regen; };
    //  CharacterAttributes*    charAttr;         //!< the character attributes of a world_entity

  void setShield(float shield) { this->shield = shield; };
  void setShieldMax(float shield) { this->shieldMax = shield; };
  void setShieldTH(float th) { this->shieldTH = th; };
  void setShieldRegen(float regen) { this->shieldRegen = regen; };
  void setShieldActive(bool active) { this->bShieldActive = active; };

  void setElectronic(float electronic) { this->electronic = electronic; };
  void setElectronicMax(float electronic) { this->electronicMax = electronic; };
  void setElectronicTH(float th) { this->electronicTH = th; };
  void setElectronicRegen(float regen) { this->electronicRegen = regen; };

  inline void drawDebugTrack(int flag) { this->bDrawTrack = (bool)flag; }
  inline bool isDrawTrack() const { return this->bDrawTrack; }

private:
  void updateHealthWidget();
  void updateElectronicWidget();
  void updateShieldWidget();
  void addTrack(const TiXmlElement* root);



protected:
  std::vector<MountPoint*> mountPoints;       //!< A list with mount points for this model
  std::map<int, MountPoint*> mountPointMap;



private:
  //!< TODO maybe we will move the following three entries and the corresponding functions to Playable AND NPC
  //!< started transfering shield/electronic/health from spaceship to WE! (nico, 4.Jun.07)
  float                   damage;             //!< the damage dealt to other objects by colliding.
  float                   health;             //!< The Energy of this Entity, if the Entity has any energy at all.
  float                   healthMax;          //!< The Maximal energy this entity can take.
  float                   healthRegen;        //!< Regeneration Rate of Health, mesured in units per second
  OrxGui::GLGuiEnergyWidgetVertical* healthWidget;    //!< The Slider (if wanted).

  float       shield;             //!< current shield
  float       shieldMax;          //!< maximum shield
  float       shieldRegen;        //!< shield regeneration rate per second
  float       shieldTH;           //!< shield threshhold for reactivation
  bool        bShieldActive;      //!< wheather the shield is working
  OrxGui::GLGuiEnergyWidgetVertical* shieldWidget; //!< holds the widget that shows the shield bar

  float       electronic;         //!< current electronic
  float       electronicMax;      //!< maximum electronic
  float       electronicRegen;    //!< electronic regenration rate per tick
  float       electronicTH;       //!< Threshhold for electronic failure
  OrxGui::GLGuiEnergyWidgetVertical* electronicWidget; //!< holds the widget that shows the electronic bar


  std::vector<Model*>     models;             //!< The model that should be loaded for this entity.
  ObjectInformationFile*  oiFile;             //!< Reference to the object information file discribing the model of this WE
  std::string             md2TextureFileName; //!< the file name of the md2 model texture, only if this
  std::string             modelLODName;       //!< the name of the model lod file
  BVTree*                 obbTree;            //!< this is the obb tree reference needed for collision detection
  AABBTreeNode*           aabbNode;           //!< the tree node of the first level of a axis aligned bounding boxes tree: model dimension

  bool                    bCollide;           //!< If it should be considered for the collisiontest.
  bool                    bVisible;           //!< If it should be visible.

  OM_LIST                 objectListNumber;                //!< The ObjectList from ObjectManager this Entity is in.
  ObjectManager::EntityList::iterator objectListIterator;  //!< The iterator position of this Entity in the given list of the ObjectManager.
  OM_LIST                 lastObjectListNumber;            //!< the last ObjectList from the ObjectManager this Entity was is in

  /* collision reaction stuff */
  CoRe::CollisionFilter   _collisionFilter;                //!< filter for collision event filtering (not every entity listens to all collisions)
  bool                    _bOnGround;                      //!< flag true if the object is on the ground

  PhysicsInterface        physicsInterface;                //!< the physics object of the WorldEntity

  /* network help structures */
  float                   scaling;                         //!< model's scaling factor
  int                     scaling_handle;                  //!< handle for syncing var

  std::string             modelFileName;                   //!< model's file name
  int                     modelFileName_handle;            //!< handle for syncing var

  int                     list_write;                      //!< entity's list
  int                     list_handle;                     //!< handle for list changes

  float                   health_write;
  int                     health_handle;

  float                   healthMax_write;
  int                     healthMax_handle;



protected:
  Vector                  velocity;                        //!< speed of the entity
  Track*                  entityTrack;                     //!< this is the track this entity follows (or NULL if none)
  bool                    bDrawTrack;                      //!< if true draws the debug track

};

#endif /* _WORLD_ENTITY_H */
