/*!
 * @file space_ship.h
 * Implements the Control of a Spaceship
 * Space Ships are the core class for all types of ships in Orxonox
 * By default is on OM_GROUP_00
 * If player boards the ship, it is moved to OM_GROUP_01
 */

#ifndef _SPACE_SHIP_H
#define _SPACE_SHIP_H

#include "playable.h"
#include "extendable.h"
#include "world_entity.h"

// Forward Declaration
template<class T> class tList;
class Vector;
class Event;
class ParticleEmitter;
class ParticleSystem;
class Trail;
class Wobblegrid;

class SpaceShip : public Playable
{
  ObjectListDeclaration(SpaceShip);

  public:
    SpaceShip(const std::string& fileName);
    SpaceShip(const TiXmlElement* root = NULL);
    virtual ~SpaceShip();

    virtual void loadParams(const TiXmlElement* root);

    virtual void setPlayDirection(const Quaternion& rot, float speed = 0.0f);
    /*
    void setTravelHeight(float travelHeight);
    void setTravelDistance(const Vector2D& distance);
    void setTravelDistance(float x, float y);
    */

    //void setAirFriction(float friction) { this->airFriction = friction; };

    virtual void enter();
    virtual void leave();

    virtual void reset();

    virtual void postSpawn();
    virtual void leftWorld();

    virtual void destroy(WorldEntity* killer);
    virtual void respawn();

    inline Vector getVelocity() { return this->velocity; };

    virtual void tick(float time);
    virtual void draw() const;

    virtual void process(const Event &event);
//    virtual void hit (WorldEntity* entity, float damage);

    inline WeaponManager& getWeaponManagerSecondary() { return this->secWeaponMan; };

    //!< functions for XML loading
    void loadReactor(float output) {this->reactorOutput = output; };
    void loadEngine( float speedBase) {this->engineSpeedBase = speedBase; };
    void loadEnergyShare(float shield, float weapon, float engine)
        { float tmp = shield + weapon + engine; if (unlikely (tmp > 1)) { tmp = 1/tmp; }
          // HACK dirty safety hack, prevents total share being bigger than 1!!
          this->shieldEnergyShare = shield * tmp;
          this->weaponEnergyShare = weapon * tmp;
          this->engineEnergyShare = engine * tmp; };
    inline void loadWeapon(float regen) { this->weaponEnergyRegen = regen; };

    void addWeaponToSlot(int wm, int config, int slot, const std::string& weaponName);

    inline PNode* getTravelNode() { return this->travelNode; };

    //damage handler
    virtual void damage(float pDamage, float eDamage);  //!< pDamage physical damage, eDamage electronic damage

    //included by Michel: 
    virtual void enterPlaymode(Playable::Playmode playmode);
    void setPlaymodeXML(const std::string& playmode); //recieves the playmode from a string (useful for script implementation)
    void setActionWidthPercentage(int i);
    void setTravelSpeed(float f);
    void updateTravelDistance();
    virtual void movement (float dt);



    void nextWeaponConfig();
    void previousWeaponConfig();

    virtual void hit(float damage, WorldEntity* killer);

    void                  setCameraDistance(float dist);

  private:
    void init();

    //void calculateVelocity(float time);

//    void regen(float time);  //!< handler for shield and electronic regeneration

    void weaponRegen(float time);   //!< weapon energy regeneration

//     inline bool systemFailure() {  return (this->getElectronic() < float(rand())/float(RAND_MAX) * this->getElectronicTH()); };

//     void updateElectronicWidget();
//     void updateShieldWidget();

    //WeaponManager         weaponMan;      //!< the primary weapon manager: managing a list of energy weapons to wepaon-slot mapping
    WeaponManager         secWeaponMan;       //!< the secondary weapon manager: managing a list of special weapons to weapon-slot mapping
    short                 supportedPlaymodes; //!< What Playmodes are Supported in this Playable.
    Playable::Playmode    playmode;           //!< The current playmode.

    //ship atributes
//    float       shieldCur;          //!< current shield
//    float       shieldMax;          //!< maximum shield
//    float       shieldRegen;        //!< shield regeneration rate per second
//    float       shieldTH;           //!< shield threshhold for reactivation
//    bool        shieldActive;       //!< wheather the shield is working
//    OrxGui::GLGuiEnergyWidgetVertical* shieldWidget; //!< holds the widget that shows the shield bar

//    float       armorCur;           //!< current armor
//    float       armorMax;           //!< maximum armor
//    float       armorRegen;         //!< armor regeneration per tick (usable on bioships?)
    //note that the armor widget is set on the health- widget in world- entity (see in player.cc)

//    float       electronicCur;      //!< current electronic
//    float       electronicMax;      //!< maximum electronic
//    float       electronicRegen;    //!< electronic regenration rate per tick
//    float       electronicTH;       //!< Threshhold for electronic failure
//    OrxGui::GLGuiEnergyWidgetVertical* electronicWidget; //!< holds the widget that shows the electronic bar

    float       engineSpeedCur;     //!< speed output for movement = speed base + energy share part
    float       engineSpeedBase;    //!< speed base
    int         enginePowerConsume; //!< energy needed
    float       engineEnergyShare;  //!< percentage of reactor output
    float       shieldEnergyShare;  //!< percentage of reactor output

    int         weaponEnergySlot;   //!< number of energy weapon slots
    int         weaponEnergyUsed;
    float       weaponEnergyShare;
    float       weaponEnergyRegen;
    int         weaponSpecialSlot;  //!< number of special weapon slots
    int         weaponSpecialUsed;

    float       reactorOutput;      //!< reactor output 
//    float       reactorCapacity;    //!< reactor capacity

    int         curWeaponPrimary;   //!< current primary weapon config 
    int         curWeaponSecondary; //!< current secondary weapon config

    bool                  bForward;           //!< up button pressed.
    bool                  bBackward;          //!< down button pressed.
    bool                  bLeft;              //!< left button pressed.
    bool                  bRight;             //!< right button pressed.
    bool                  bAscend;            //!< ascend button pressed.
    bool                  bDescend;           //!< descend button presses.
    bool                  bRollL;             //!< rolling button pressed (left)
    bool                  bRollR;             //!< rolling button pressed (right)
    bool                  bSecFire;           //!< second fire button pressed

    bool                  bInit;              //!< set true, if MP have been loaded

    /*
    float                 xMouse;             //!< mouse moved in x-Direction
    float                 yMouse;             //!< mouse moved in y-Direction
    float                 mouseSensitivity;   //!< the mouse sensitivity
    int                   yInvert;
    int                   controlVelocityX;
    int                   controlVelocityY;
    */

    Vector                velocity;           //!< the velocity of the player.
    Vector                oldPos;

// 2D-traveling
    PNode*                travelNode;
    float                 travelSpeed;        //!< the current speed of the Ship (to make soft movement)
    Vector                travelVelocity;     //!< object internal velocity vector for relative movement to the track node
    Vector2D              travelDistancePlus;     //!< Travel-Distance away from the TravelNode
    Vector2D              travelDistanceMinus;
    bool                  isTravelDistanceInit;

    float                 actionWidthPercentage;

// Camera
    PNode                 cameraNode;
    float                 cameraLook;
    float                 rotation;
    float                 cameraSpeed;


    void                  setCameraFovy(float fovy);

    /*
    Quaternion            mouseDir;           //!< the direction where the player wants to fly
    Quaternion            oldMouseDir;        //!< the direction where the player wanted to fly
    float                 shipInertia;        //!< the inertia of the ship(how fast the ship reacts to a mouse input)
    Quaternion            rotQuat;
    Quaternion            pitchDir;
    float                 dogdeSpeed;        //!< the dogde Speed of the ship.
    */

    //Quaternion            direction;          //!< the direction of the ship.
    float                 acceleration;       //!< the acceleration of the ship.
    //float                 airFriction;        //!< AirFriction.
    //float                 airViscosity;

    byte                  oldMask;            //!< used for synchronisation
/*
    Trail*                trail;              //!< Burst trail
    Trail*                trailL;              //!< Burst trail
    Trail*                trailR;              //!< Burst trail
*/

};

#endif /* _SPACE_SHIPS_H */
