/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Knecht
   co-programmer: Silvan Nellen

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "executor/executor.h"
#include "space_ship.h"

#include "util/loading/resource_manager.h"

#include "weapons/test_gun.h"
#include "weapons/light_blaster.h"
#include "weapons/medium_blaster.h"
#include "weapons/heavy_blaster.h"
#include "weapons/rf_cannon.h"
#include "weapons/nadion_laser.h"
#include "weapons/disruptor.h"
#include "weapons/swarm_launcher.h"
#include "weapons/spike_thrower.h"
#include "weapons/acid_launcher.h"
#include "weapons/boomerang_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "elements/glgui_energywidgetvertical.h"
#include "glgui_bar.h"

#include "particles/dot_emitter.h"
#include "particles/emitter_node.h"
#include "particles/sprite_particles.h"
#include "effects/trail.h"

#include "effects/wobblegrid.h"

#include "util/loading/factory.h"
#include "key_mapper.h"

#include "network_game_manager.h"
#include "shared_network_data.h"

#include "items/power_ups/weapon_power_up.h"
#include "items/power_ups/param_power_up.h"

#include "graphics_engine.h"

#include "plane.h"

#include "state.h"
#include "player.h"
#include "tools/camera.h"
#include "tools/cameraman.h"


#include "util/loading/load_param.h"
#include "time.h"

#include "track/track.h"
#include "track/action_box.h"


ObjectListDefinition(SpaceShip);
CREATE_FACTORY(SpaceShip);

#include "script_class.h"
CREATE_SCRIPTABLE_CLASS(SpaceShip,
                        addMethod("hasPlayer", Executor0ret<Playable, lua_State*,bool>(&Playable::hasPlayer))
                        ->addMethod("fire", Executor1<Playable, lua_State*, bool>(&Playable::fire))
                        ->addMethod("loadModel", Executor2<WorldEntity, lua_State*,const std::string& ,float>(&WorldEntity::loadModel2))
                        ->addMethod("setName", Executor1<BaseObject, lua_State*,const std::string&>(&BaseObject::setName))
                        ->addMethod("hide", Executor0<WorldEntity, lua_State*>(&WorldEntity::hide))
                        ->addMethod("unhide", Executor0<WorldEntity, lua_State*>(&WorldEntity::unhide))
                        //Coordinates
                        ->addMethod("setAbsCoor", Executor3<PNode, lua_State*,float,float,float>(&PNode::setAbsCoor))
                        ->addMethod("getAbsCoorX", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorX))
                        ->addMethod("getAbsCoorY", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorY))
                        ->addMethod("getAbsCoorZ", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorZ))
                        //->addMethod("setCameraSpeed", Executor1<SpaceShip, lua_State*, float>(&SpaceShip::setCameraSpeed))
                        ->addMethod("pause", Executor1<WorldEntity, lua_State*, bool>(&WorldEntity::pauseTrack))
                        ->addMethod("setCameraDist", Executor1<SpaceShip, lua_State*, float>(&SpaceShip::setCameraDistance))
                       );

/**
 *  destructs the spaceship, deletes alocated memory
 */
SpaceShip::~SpaceShip ()
{
  this->setPlayer(NULL);
}

/**
 * loads a Spaceships information from a specified file.
 * @param fileName the name of the File to load the spaceship from (absolute path)
 */
SpaceShip::SpaceShip(const std::string& fileName)
    : secWeaponMan(this) //,
    //supportedPlaymodes(Playable::Vertical) ,
    //playmode(Playable::Vertical)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for spaceship.\n", fileName.c_str());
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 *  creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
SpaceShip::SpaceShip(const TiXmlElement* root)
    : secWeaponMan(this) //,
    //supportedPlaymodes(Playable::Vertical) ,
    //playmode(Playable::Vertical)
{
  this->init();
  //this->setParentMode(PNODE_REPARENT_DELETE_CHILDREN);
  if (root != NULL)
    this->loadParams(root);

}


/**
 * initializes a Spaceship
 */
void SpaceShip::init()
{

  srand(time(0));   //initialize Random Nomber Generator

  //  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->registerObject(this, SpaceShip::_objectList);
  PRINTF(4)("SPACESHIP INIT\n");
  this->weaponMan.setParentEntity( this);
  //weapons:

  this->weaponMan.setParentEntity( this);
  this->secWeaponMan.setParentEntity( this);

  this->setWMSlotCount(1,8);
  this->setWMSlotCount(2,6);

  this->createWeaponSlot(1, 0, Vector(3.006, 1.028, .155), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 1, Vector(3.006, 1.028, -.155), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 2, Vector(4.03, .063, .876), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 3, Vector(4.03, -.063, -.876), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 4, Vector(1.431, -.612, 2.691), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 5, Vector(1.431, -.612, -2.691), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 6, Vector(1.431, -.612, 3.254), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(1, 7, Vector(1.431, -.612, -3.254), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);


  this->addWeaponToSlot(1, 0, 0, "RFCannon");
  this->addWeaponToSlot(1, 0, 1, "RFCannon");
  this->addWeaponToSlot(1, 0, 2, "RFCannon");
  this->addWeaponToSlot(1, 0, 3, "RFCannon");
  this->addWeaponToSlot(1, 1, 0, "RFCannon");
  this->addWeaponToSlot(1, 1, 1, "RFCannon");
  this->addWeaponToSlot(1, 1, 2, "RFCannon");
  this->addWeaponToSlot(1, 1, 3, "RFCannon");

  this->addWeaponToSlot(1, 0, 4, "NadionLaser");
  this->addWeaponToSlot(1, 0, 5, "NadionLaser");
  this->addWeaponToSlot(1, 2, 4, "NadionLaser");
  this->addWeaponToSlot(1, 2, 5, "NadionLaser");

  this->addWeaponToSlot(1, 0, 6, "Disruptor");
  this->addWeaponToSlot(1, 0, 7, "Disruptor");
  this->addWeaponToSlot(1, 3, 6, "Disruptor");
  this->addWeaponToSlot(1, 3, 7, "Disruptor");


  this->createWeaponSlot(2, 0, Vector(1.5, 3, 0), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(2, 1, Vector(2.6, 0, 3.0), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(2, 2, Vector(1.5, 0, -.5), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(2, 3, Vector(1.5, 0, .5), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(2, 4, Vector(1.5, 0, .5), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
  this->createWeaponSlot(2, 5, Vector(1.5, 0, -.5), WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->addWeaponToSlot(2, 0, 2, "SwarmLauncher");


  this->weaponMan.changeWeaponConfig(0);
  this->secWeaponMan.changeWeaponConfig(0);


  Playable::weaponConfigChanged();

  setReactor(10);
  setWeapon(10);
  setEngine(15);


  setEnergyShare(.3,.3,.4);
  setShield(100, 100, .2, 2);
  setArmor(100, 100);
  setElectronic(50, 50, .7, 3.0);

//   this->loadModel("models/spaceships/fighter_redesign9.obj");
  //this->setVisibiliy(false);

  bForward = bBackward = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = bFire = bSecFire = false;

//   this->setHealthMax(shieldMax);
//   this->setHealth(shieldCur);

  this->travelNode = new PNode();

  // camera - issue
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);

  // widget handling
  /*
  this->electronicWidget = new OrxGui::GLGuiEnergyWidgetVertical();
  this->electronicWidget->setDisplayedName(std::string(this->getClassName()) + " Electronics:");
  this->electronicWidget->setSize2D(30,400);
  this->electronicWidget->setAbsCoor2D(150,200);
  this->electronicWidget->shiftDir2D(270);
  this->updateElectronicWidget();
  this->shieldWidget = new OrxGui::GLGuiEnergyWidgetVertical();
  this->shieldWidget->setDisplayedName(std::string(this->getClassName()) + " Shield:");
  this->shieldWidget->setSize2D(30,400);
  this->shieldWidget->setAbsCoor2D(200,200);
  this->shieldWidget->shiftDir2D(270);
  this->updateShieldWidget();
  if (this->hasPlayer())
  {
    State::getPlayer()->hud().setShiledWidget(this->shieldWidget);
    State::getPlayer()->hud().setEnergyWidget(this->electronicWidget);
  }
  */
  this->electronicWidget = NULL;
  this->shieldWidget = NULL;

  //add events to the eventlist
  registerEvent(KeyMapper::PEV_FORWARD);
  registerEvent(KeyMapper::PEV_BACKWARD);
  registerEvent(KeyMapper::PEV_LEFT);
  registerEvent(KeyMapper::PEV_RIGHT);
  //registerEvent(SDLK_q);
  //registerEvent(SDLK_e);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_FIRE2);                  // Added for secondary weapon support
  registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  //registerEvent(SDLK_PAGEUP);
  //registerEvent(SDLK_PAGEDOWN);
  registerEvent(EV_MOUSE_MOTION);


  this->weaponMan.getFixedTarget()->setParent(this);
  this->weaponMan.getFixedTarget()->setRelCoor(100000,0,0);


  this->secWeaponMan.getFixedTarget()->setParent(this);
  this->secWeaponMan.getFixedTarget()->setRelCoor(100000,0,0);
  this->secWeaponMan.setRotationSpeed(0);

  dynamic_cast<Element2D*>(this->weaponMan.getFixedTarget())->setVisibility( false);


  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &cameraLook, &cameraLook, "cameraLook", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &rotation, &rotation, "rotation", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bFire, &bFire, "bSecFire", PERMISSION_OWNER));
  registerVar( new SynchronizeableVector( &velocity, &velocity, "velocity", PERMISSION_MASTER_SERVER ) );

  //this->airFriction = 0.5f;
  //this->travelDistancePlus = Vector2D(38.0, 43.0);
  //this->travelDistanceMinus = Vector2D(-38.0, -43.0);
  this->travelDistancePlus = Vector2D(50,50);
  this->travelDistanceMinus = Vector2D(-50,-50);
  this->isTravelDistanceInit = true;
  this->actionWidthPercentage = 1;

  this->cameraSpeed = 40;
  this->cameraLook = 0.0f;
  //this->airFriction = 0.0f;

  srand(time(0));  //initaialize RNG

  this->travelNode->debugDraw();

  this->setSupportedPlaymodes(Playable::Horizontal | Playable::Vertical);

  /// FIXME
//   this->trail = new Trail( 5, 10, .2, this);
//   this->trail->setTexture( "textures/engine.png");
//
//   this->trailL = new Trail( 5, 10, .2, this);
//   this->trailL->setTexture( "textures/engine.png");
//
//   this->trailR = new ( 5, 10, .2, this);
//   this->trailR->setTexture( "textures/engine.png");


  this->toList(OM_GROUP_00);

  //FIXME Just testaddition to show the wobblegrid
/*
  this->test  = new Wobblegrid(5);
  test->setTexture("textures/blub.png");

  test->setAbsCoor( this->getAbsCoor() + Vector(0, 2, 0));
  test->setParent( this);
*/

}


/**
 * loads the Settings of a SpaceShip from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void SpaceShip::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);

  LoadParam(root, "playmode", this, SpaceShip, setPlaymodeXML);
  LoadParam(root, "cameraDistance", this, SpaceShip, setCameraDistance);
  LoadParam(root, "cameraFovy", this, SpaceShip, setCameraFovy);
  LoadParam(root, "actionWidthPercentage", this, SpaceShip, setActionWidthPercentage);

  State::getCamera()->setViewMode(Camera::ViewNormal);
  State::getCameraTargetNode()->setParent(this);
  State::getCamera()->setParent(this);

  LoadParam(root, "reactor", this, SpaceShip, setReactor);
  LoadParam(root, "shield", this, SpaceShip, setShield);
  LoadParam(root, "armor", this, SpaceShip, setArmor);
  LoadParam(root, "electronic", this, SpaceShip, setElectronic);
  LoadParam(root, "engine", this, SpaceShip, setEngine);
  LoadParam(root, "energyshare", this, SpaceShip, setEnergyShare);
  LoadParam(root, "weapon", this, SpaceShip, setWeapon);

}


void SpaceShip::setPlayDirection(const Quaternion& rot, float speed)
{
  //this->direction = Quaternion (rot.getHeading(), Vector(0,1,0));
}

void SpaceShip::reset()
{
  bForward = bBackward = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = bFire = bSecFire = false;

  //xMouse = yMouse = 0;

  this->setHealth(80);
  this->velocity = Vector(0.0, 0.0, 0.0);
}


void SpaceShip::enter()
{
  this->secWeaponMan.showCrosshair();
  this->toList( OM_GROUP_01 );
  State::getPlayer()->hud().setRadarCenterNode(this->travelNode);
  State::getPlayer()->hud().setOverlayActive(true);
  //dynamic_cast <OrxGui::GLGuiEnergyWidgetVertical*> (State::getPlayer()->hud().getArmorWidget())->setDisplayedName("Armor");
  //dynamic_cast<Element2D*>(this->secWeaponMan.getFixedTarget())->setVisibility( true);
  //this->attachCamera();
 // this->setPlaymode(Playable::Horizontal);
}

void SpaceShip::leave()
{
  this->secWeaponMan.hideCrosshair();
  this->toList( OM_GROUP_00);
  State::getPlayer()->hud().setOverlayActive(false);
  State::getCamera()->setEventHandling(true);
  State::getPlayer()->hud().setRadarCenterNode(NULL);
  //dynamic_cast<Element2D*>(this->secWeaponMan.getFixedTarget())->setVisibility( false);
  //this->detachCamera();
}


/**
 *  effect that occurs after the SpaceShip is spawned
*/
void SpaceShip::postSpawn ()
{
  if(this->hasPlayer())
    Playable::postSpawn();

  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the spaceship left the game
*/
void SpaceShip::leftWorld ()
{

}

WorldEntity* ref = NULL;

/**
 *  draws the spaceship after transforming it.
*/
void SpaceShip::draw () const
{
  if( this->entityTrack != NULL && this->isDrawTrack())
    this->entityTrack->drawGraph();

  WorldEntity::draw();

//   glMatrixMode(GL_MODELVIEW);
//   glPushMatrix();

//   float matrix[4][4];
//   glTranslatef (this->getAbsCoor ().x-1, this->getAbsCoor ().y-.2, this->getAbsCoor ().z);
//   this->getAbsDir().matrix (matrix);
//   glMultMatrixf((float*)matrix);
  //glScalef(2.0, 2.0, 2.0);  // no double rescale


//   this->trail->draw();

//   glTranslatef(0,0,-.5);
//   this->trailL->draw();

//   glTranslatef(0,0,1);
//   this->trailR->draw();

//   glPopMatrix();
  //this->debug(0);
}

/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void SpaceShip::tick (float time)
{
  // Playable::tick(time);

 // this->test->tick(time);

  // Own Tick Setup, as a different fire routine is used on the weapon manager
  this->weaponMan.tick(time);
  this->secWeaponMan.tick(time);

  if( this->systemFailure() )
    bFire = bSecFire = false;

  // fire reqeust/release for primary weapons
  if( this->bFire)
    this->weaponMan.fire();
  else
    this->weaponMan.releaseFire();

  // fire reqeust/release for secondary weapons
  if( this->bSecFire)
    this->secWeaponMan.fire();
  else
    this->secWeaponMan.releaseFire();

  // Tracktick
  if(this->entityTrack)
    this->entityTrack->tick(time);


  // Shield Regeneration and other regular calculations on the ship
  this->regen(time);

  // Weapon Regeneration and other regular calculations on the ship
  this->weaponRegen(time);

  // current engine speed output
  this->engineSpeedCur = this->engineSpeedBase + this->reactorOutput * this->engineEnergyShare;

  // calculation of maxSpeed and acceleration:
  this->travelSpeed = this->engineSpeedCur * 5;
  this->acceleration = this->travelSpeed * 2;

  this->movement(time);

   // TRYING TO FIX PNode.
  this->cameraNode.setAbsCoorSoft(this->getAbsCoor() + Vector(0.0f, 5.0f, 0.0f), 30.0f);
  this->cameraNode.setRelDirSoft(this->getAbsDir(), 30.0f);


  this->velocity  = (this->getAbsCoor() - this->oldPos) / time;
  this->oldPos    = this->getAbsCoor();

//FIXME
//   this->trail->tick(time);
//   this->trailL->tick(time);
//   this->trailR->tick(time);

  if (!this->isTravelDistanceInit)
  {
    this->updateTravelDistance();
    //this->isTravelDistanceInit = true;
  }

  //orient the spaceship in direction of the mouse
  /*
  rotQuat = Quaternion::quatSlerp( this->getAbsDir(), mouseDir, 0.5);//fabsf(time)*shipInertia);
  if (this->getAbsDir().distance(rotQuat) > 0.00000000000001)
    this->setAbsDir( rotQuat);
  //this->setAbsDirSoft(mouseDir,5);
  */
  /*
  this->shiftCoor(move);
  */

  //   PRINTF(0)("id of %s is: %i\n", this->getName(), this->getOMListNumber());

}

/**
 * @todo switch statement ??
 */
void SpaceShip::process(const Event &event)
{
  //Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
  {
    this->bRight = event.bPressed;
//     printf("ShipCoors: %f , %f, %f \n", this->getAbsCoor().x, this->getAbsCoor().y, this->getAbsCoor().z);
  }
  else if( event.type == KeyMapper::PEV_FORWARD)
  {
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);

  }
  else if( event.type == KeyMapper::PEV_BACKWARD)
    this->bBackward = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == KeyMapper::PEV_FIRE2)
    this->bSecFire = event.bPressed;
  else if( event.type == KeyMapper::PEV_FIRE1)
    this->bFire = event.bPressed;
  else if( event.type == KeyMapper::PEV_NEXT_WEAPON && event.bPressed)
  {
    this->nextWeaponConfig();
  }
  else if ( event.type == KeyMapper::PEV_PREVIOUS_WEAPON && event.bPressed)
    this->previousWeaponConfig();

  if (!(State::getCamera()->getEventHandling()))
  {
    //PRINTF(0)("\n\n\n\n\n\n\n\nSETCAMERA %x\n\n\n\n\n\n\n", State::getCamera());
    if( event.type == KeyMapper::PEV_VIEW0)
    {
      State::getCamera()->setViewMode(Camera::ViewNormal);
      State::getCameraTargetNode()->setParent(this);
      State::getCamera()->setParent(this);
    }
    else if( event.type == KeyMapper::PEV_VIEW1)
    {
      State::getCamera()->setViewMode(Camera::ViewBehind);
      State::getCameraTargetNode()->setParent(this);
      State::getCamera()->setParent(this);
    }
    else if( event.type == KeyMapper::PEV_VIEW2)
    {
      State::getCamera()->setViewMode(Camera::ViewFront);
      State::getCameraTargetNode()->setParent(this);
      State::getCamera()->setParent(this);
    }
    else if( event.type == KeyMapper::PEV_VIEW3)
    {
      State::getCamera()->setViewMode(Camera::ViewLeft);
      State::getCameraTargetNode()->setParent(this);
      State::getCamera()->setParent(this);
    }
    else if( event.type == KeyMapper::PEV_VIEW4)
    {
      State::getCamera()->setViewMode(Camera::ViewRight);
      State::getCameraTargetNode()->setParent(this);
      State::getCamera()->setParent(this);
    }
    else if( event.type == KeyMapper::PEV_VIEW5)
    {
      State::getCamera()->setViewMode(Camera::ViewTop);
      State::getCameraTargetNode()->setParent(this->travelNode);
      State::getCamera()->setParent(this->travelNode);
    }
  }


  /*
  else if( event.type == EV_MOUSE_MOTION)
  {

    this->xMouse += event.xRel;
    this->yMouse += event.yRel;
  }
  */
}

void SpaceShip::destroy( WorldEntity* killer )
{
  if(this->hasPlayer())
    Playable::destroy( killer);

  PRINTF(5)("spaceship destroy\n");

  EmitterNode* node  = NULL;
  DotEmitter* emitter = NULL;
  SpriteParticles*  explosionParticles  = NULL;

  explosionParticles = new SpriteParticles(200);
  explosionParticles->setName("SpaceShipExplosionParticles");
  explosionParticles->setLifeSpan(.2, .3);
  explosionParticles->setRadius(0.0, 10.0);
  explosionParticles->setRadius(.5, 6.0);
  explosionParticles->setRadius(1.0, 3.0);
  explosionParticles->setColor(0.0, 1,1,1,.9);
  explosionParticles->setColor(0.1,  1,1,0,.9);
  explosionParticles->setColor(0.5, .8,.4,0,.5);
  explosionParticles->setColor(1.0, .2,.2,.2,.5);


  emitter = new DotEmitter( 2000, 70, 360);
  //emitter->setSpread( 0, M_2_PI);
  emitter->setEmissionRate( 200.0);
  //emitter->setEmissionVelocity( 200.0);
  //emitter->setSystem( explosionParticles);
  //emitter->setAbsCoor( this->getAbsCoor());

  node  = new EmitterNode( .1f);
  node->setupParticle( emitter, explosionParticles);
  node->setAbsDir( this->getAbsDir());
  node->setVelocity( this->getVelocity() * .9f);
  node->setAbsCoor( this->getAbsCoor());
  if( !node->start())
    PRINTF(0)("Explosion node not correctly started!");
/*
  PNode* node          = new PNode();
  node->setAbsCoor(this->getAbsCoor());
  Explosion* explosion = new Explosion();
  explosion->explode( node, Vector(5,5,5));
*/
/*
  if( this->hasPlayer())
  {
	this->setAbsCoor(Vector(-10000,10000,10000));
	this->hide();
  }
  else
  {*/
    this->setAbsCoor( this->getAbsCoor() + Vector(100,0,0) + Vector(1,0,0) * VECTOR_RAND(150).dot(Vector(1,0,0)));
  //}

}

void SpaceShip::respawn( )
{
  Playable::respawn();
}


void SpaceShip::damage(float pDamage, float eDamage){
  PRINTF(5)("ship hit for (%f,%f) \n",pDamage,eDamage);

  if( this->shieldActive) {
    if( this->shieldCur > pDamage) {
      this->shieldCur = this->shieldCur - pDamage;
    }
    else { // shield <= pDamage
      this->shieldCur -=pDamage;
      this->shieldActive = false; //shield collapses
      pDamage += this->shieldCur;
      if( !this->shieldActive) {
        this->armorCur -= pDamage / 2; // remaining damages hits armor at half rate
        this->electronicCur -= eDamage;
      }
    }
  }
  else {
    this->armorCur = this->armorCur - pDamage;
    this->electronicCur = this->electronicCur - eDamage;
  }
  if( this->armorCur <= 0) { /* FIXME implement shipcrash*/ }
    this->destroy(this);

  updateElectronicWidget();
  updateShieldWidget();

  this->setHealth( this->armorCur);
}


void SpaceShip::regen(float time){
  float tmp;
  if (this->armorCur != this->armorMax || this->armorRegen != 0){
    tmp = this->armorCur + this->armorRegen * time;
    if ( tmp > electronicMax)
      this->armorCur = this->armorMax;
    else
      this->armorCur = tmp;
  }
  if (this->shieldCur != this->shieldMax || this->shieldRegen != 0){
    tmp =  this->shieldCur + (this->shieldRegen + this->reactorOutput * this->shieldEnergyShare) * time;
    if( tmp > shieldMax)
      this->shieldCur = this->shieldMax;
    else
      this->shieldCur = tmp;
    this->shieldActive = ( this->shieldActive || this->shieldCur > shieldTH);

    updateShieldWidget();
  }

  this->setHealth( this->shieldCur);      // FIXME currently just to test share system

  if (this->electronicCur != this->electronicMax || this->electronicRegen != 0){
    tmp = this->electronicCur + this->electronicRegen * time;
    if ( tmp > electronicMax)
      this->electronicCur = this->electronicMax;
    else
      this->electronicCur = tmp;

    updateElectronicWidget();
  }

}


/**
 * Weapon regeneration
 * does not use any reactor capacity, as it wouldn't work in a consistent way.
 */
void SpaceShip::weaponRegen(float time)
{
  float energy  = ( this->reactorOutput * this->weaponEnergyShare + this->weaponEnergyRegen) * time;
  Weapon* weapon;
  for( unsigned int i=0; i < this->weaponMan.getSlotCount(); i++)
  {
    weapon = this->weaponMan.getWeapon(i);
    if( weapon != NULL && weapon->isActive())
    {
      weapon->increaseEnergy( energy);
    }

  }
  // weaponMan.increaseAmmunition( weapon, energy);
}


void SpaceShip::enterPlaymode(Playable::Playmode playmode)
{
  switch(playmode)
  {
    case Playable::Full3D:
      /*
      if (State::getCameraNode != NULL)
      {
        Vector absCoor = this->getAbsCoor();
        this->setParent(PNode::getNullParent());
        this->setAbsCoor(absCoor);
        State::getCameraNode()->setParentSoft(&this->cameraNode);
        State::getCameraNode()->setRelCoorSoft(-10, 0,0);
        State::getCameraTargetNode()->setParentSoft(&this->cameraNode);
        State::getCameraTargetNode()->setRelCoorSoft(100, 0,0);

      }
      */
      //break;

      break;
    case Playable::Horizontal:
      if (State::getCameraNode != NULL)
      {
        this->debugNode(1);
        this->travelNode->debugNode(1);

        this->travelNode->setAbsCoor(this->getAbsCoor());
        this->travelNode->updateNode(0.01f);

        this->isTravelDistanceInit = false;

        if(this->entityTrack)
           this->travelNode->setParent(this->entityTrack->getTrackNode());

        this->setParent(this->travelNode);
        this->setRelCoor(0,0,0);

        State::getCameraNode()->setParentSoft(this->travelNode);
        //State::getCameraNode()->setParentSoft(this);
        //State::getCameraNode()->setRelCoorSoft(-0.01, 40, 0);
        State::getCameraTargetNode()->setParentSoft(this->travelNode);
        //State::getCameraTargetNode()->setParentSoft(this);
        //State::getCameraTargetNode()->setRelCoorSoft(0,0,0);
        this->setCameraMode(Camera::ViewTop);
        State::getCamera()->setEventHandling(false);
        registerEvent(KeyMapper::PEV_VIEW0);
        registerEvent(KeyMapper::PEV_VIEW1);
        registerEvent(KeyMapper::PEV_VIEW2);
        registerEvent(KeyMapper::PEV_VIEW3);
        registerEvent(KeyMapper::PEV_VIEW4);
        registerEvent(KeyMapper::PEV_VIEW5);

        State::getCamera()->setParentMode(PNODE_ALL);

        //this->updateTravelDistance();

        this->debugNode(1);
        this->travelNode->debugNode(1);
      }
      break;

    
    case Playable::Vertical:
    {
        this->travelNode->setAbsCoor(this->getAbsCoor());
        this->travelNode->updateNode(0.01f);

        this->isTravelDistanceInit = false;

        if(this->entityTrack)
           this->travelNode->setParent(this->entityTrack->getTrackNode());

        this->setParent(this->travelNode);
        this->setRelCoor(0,0,0);

        State::getCameraNode()->setParentSoft(this->travelNode);
        //State::getCameraNode()->setParentSoft(this);
        //State::getCameraNode()->setRelCoorSoft(-0.01, 40, 0);
        State::getCameraTargetNode()->setParentSoft(this->travelNode);
        //State::getCameraTargetNode()->setParentSoft(this);
        //State::getCameraTargetNode()->setRelCoorSoft(0,0,0);
        //this->setCameraMode(Camera::ViewNormal);
        State::getCamera()->setEventHandling(false);
        
        PRINTF(0)("\n\n\n\n\n\n\n\nSETCAMERA %x\n\n\n\n\n\n\n", State::getCamera());
        State::getCamera()->setViewMode(Camera::ViewNormal);
        State::getCameraTargetNode()->setParent(this);
        State::getCamera()->setParent(this);

        
        
        registerEvent(KeyMapper::PEV_VIEW0);
        registerEvent(KeyMapper::PEV_VIEW1);
        registerEvent(KeyMapper::PEV_VIEW2);
        registerEvent(KeyMapper::PEV_VIEW3);
        registerEvent(KeyMapper::PEV_VIEW4);
        registerEvent(KeyMapper::PEV_VIEW5);

        State::getCamera()->setParentMode(PNODE_ALL);
      
      break;
    }
      
    default:
      PRINTF(2)("Playmode %s Not Implemented in %s\n", Playable::playmodeToString(this->getPlaymode()).c_str(), this->getClassCName());
  }
}

/**
 * @brief calculate the velocity
 * @param time the timeslice since the last frame
*/

void SpaceShip::movement (float dt)
{
  //by releasing the buttons, the velocity decreases with airCoeff*Acceleration. Should be strong enough so
  //the ship doesn't slide too much.
  float airCoeff = 2.5;
  float pi = 3.14;

  switch(this->getPlaymode())
  {
    case Playable::Horizontal:
    {
      // these routines will change the travel movement into zero in a short amout of time, if the player
      // doesn't press any buttons.
      if (this->travelVelocity.x >= 0)
      {
        if (this->travelVelocity.x > airCoeff*this->acceleration * dt)
          this->travelVelocity.x -= airCoeff* this->acceleration * dt;
        else
          this->travelVelocity.x = 0;
      }
      else
      {
        if (this->travelVelocity.x < -airCoeff*this->acceleration * dt)
          this->travelVelocity.x += airCoeff* this->acceleration * dt;
        else
          this->travelVelocity.x = 0;
      }
      if (this->travelVelocity.z >= 0)
      {
        if (this->travelVelocity.z > airCoeff*this->acceleration * dt)
          this->travelVelocity.z -= airCoeff* this->acceleration * dt;
        else
          this->travelVelocity.z = 0;
      }
      else
      {
        if (this->travelVelocity.z < -airCoeff*this->acceleration * dt)
          this->travelVelocity.z += airCoeff* this->acceleration * dt;
        else
          this->travelVelocity.z = 0;
      }

      // this will evite, that the ship moves outside the travelDistance- borders,when the player releases all buttons
      // and its continuing to slide a bit.
      Vector oldCoor = this->getRelCoor();
      if (this->getRelCoor().x > this->travelDistancePlus.x) this->setRelCoor(this->travelDistancePlus.x, oldCoor.y, oldCoor.z);
      if (this->getRelCoor().x < this->travelDistanceMinus.x) this->setRelCoor(this->travelDistanceMinus.x, oldCoor.y, oldCoor.z);
      if (this->getRelCoor().z > this->travelDistancePlus.y) this->setRelCoor(oldCoor.x, oldCoor.y, this->travelDistancePlus.y);
      if (this->getRelCoor().z < this->travelDistanceMinus.y) this->setRelCoor(oldCoor.x, oldCoor.y, this->travelDistanceMinus.y);

      if( this->systemFailure() )
        bForward = bBackward = bLeft = bRight = false;

      if( this->bForward )
      {
        //printf("ShipCoorX: %f \n", this->getRelCoor().x);
        if(this->getRelCoor().x < this->travelDistancePlus.x)
        {
          if (this->travelVelocity.x < this->travelSpeed)
          {
            this->travelVelocity.x += (airCoeff + 1.0)*this->acceleration*dt;
          }
          else
          {
            this->travelVelocity.x = this->travelSpeed;
          }
        }
        else
        {
          this->travelVelocity.x = 0.0f;
        }
      }

      if( this->bBackward )
      {
        if(this->getRelCoor().x > this->travelDistanceMinus.x)
        {
          if (this->travelVelocity.x > -this->travelSpeed)
          {
            this->travelVelocity.x -= (airCoeff + 1.0)*this->acceleration*dt;
          }
          else
          {
            this->travelVelocity.x = -this->travelSpeed;
          }
        }
        else
        {
          this->travelVelocity.x = 0.0f;
        }
      }

      if( this->bLeft)
      {
        if(this->getRelCoor().z > this->travelDistanceMinus.y)
        {
          if (this->travelVelocity.z > -this->travelSpeed)
          {
            this->travelVelocity.z -= (airCoeff + 1.0)*this->acceleration*dt;
          }
          else
          {
            this->travelVelocity.z = -this->travelSpeed;
          }
        }
        else
        {
          this->travelVelocity.z = 0.0f;
        }
        this->setRelDirSoft(Quaternion(-pi/6, Vector(1,0,0)), 6);
      }

      if( this->bRight)
      {
        //printf("ShipCoorZ: %f \n", this->getRelCoor().z);
        if(this->getRelCoor().z < this->travelDistancePlus.y)
        {
          if (this->travelVelocity.z < this->travelSpeed)
          {
            this->travelVelocity.z += (airCoeff + 1.0)*this->acceleration*dt;
          }
          else
          {
            this->travelVelocity.z = this->travelSpeed;
          }
        }
        else
        {
          this->travelVelocity.z = 0.0f;
        }
        this->setRelDirSoft(Quaternion(pi/6, Vector(1,0,0)), 6);
      }
      if (!this->bRight && !this->bLeft)
      {
        this->setRelDirSoft(Quaternion(0, Vector(1,0,0)), 6);
      }

    //normalisation of the vectors (vector sum must be <= travelspeed)
    float xzNorm = sqrt(pow(this->travelVelocity.x, 2) + pow(this->travelVelocity.z, 2));
    if (xzNorm > this->travelSpeed)
    {
      this->travelVelocity.x = this->travelVelocity.x/xzNorm * this->travelSpeed;
      this->travelVelocity.z = this->travelVelocity.z/xzNorm * this->travelSpeed;
    }

    //this moves camera and ship along the travel path.
    if(!this->entityTrack)
       this->travelNode->shiftCoor(Vector(this->cameraSpeed * dt, 0, 0));

    break;
    }
    case Playable::Vertical:
    {
      if ( !entityTrack || !entityTrack->getActionBox() )
      {
        break;
      }
      ActionBox * box = entityTrack->getActionBox();
      
      this->travelVelocity = Vector(0, 0, 0);
      float ssss = 50;
      if ( this->bForward )
      {
        this->travelVelocity += Vector( 0, ssss, 0 );
      }
      if ( this->bBackward )
      {
        this->travelVelocity += Vector( 0, -ssss, 0 );
      }
      if ( this->bLeft )
      {
        this->travelVelocity += Vector( 0, 0, -ssss );
      }
      if ( this->bRight )
      {
        this->travelVelocity += Vector( 0, 0, ssss );
      }
      
      Vector ds = this->travelVelocity*dt;
      Vector newPos = this->getRelCoor() + ds;
      if ( newPos.y < -(box->getHeight_2()) || newPos.y > box->getHeight_2() )
      {
        this->travelVelocity.y = 0;
      }
      if ( newPos.z > box->getWidth_2() || newPos.z < -(box->getWidth_2()) )
      {
        this->travelVelocity.z = 0;
      }
    }
      break;
    default:
      PRINTF(4)("Playmode %s Not Implemented in %s\n", Playable::playmodeToString(this->getPlaymode()).c_str(), this->getClassCName());
  }
   //set new coordinates calculated through key- events.
  this->shiftCoor (this->travelVelocity * dt);
}

void SpaceShip::setPlaymodeXML(const std::string& playmode)
{
  this->setPlaymode(Playable::stringToPlaymode(playmode));
}

/**
 * @brief jumps to the next WeaponConfiguration
 */
void SpaceShip::nextWeaponConfig()
{
  PRINTF(0)("Requested next weapon config!\n");
  this->weaponMan.nextWeaponConfig();
  Playable::weaponConfigChanged();
}

/**
 * @brief moves to the last WeaponConfiguration
 */
void SpaceShip::previousWeaponConfig()
{
  /*
  this->curWeaponPrimary    = (this->curWeaponPrimary + 1) % 3;
  this->weaponMan.changeWeaponConfig(this->curWeaponPrimary);
  */
  this->weaponMan.previousWeaponConfig();
  Playable::weaponConfigChanged();
}

void SpaceShip::hit( float damage, WorldEntity* killer)
{
  this->damage(killer->getDamage(),0);
}

void SpaceShip::updateElectronicWidget()
{
  if (this->electronicWidget != NULL)
  { //if it exists already: update it
     this->electronicWidget->setMaximum(this->electronicMax);
     this->electronicWidget->setValue(this->electronicCur);
  }
  else
  { //create the widget
    this->electronicWidget = new OrxGui::GLGuiEnergyWidgetVertical();
    this->electronicWidget->getBarWidget()->setChangedValueColor(Color(1,0,0,1));
    //this->electronicWidget->setDisplayedName("Electronics:");
    //this->electronicWidget->setSize2D(100,20);
    //this->electronicWidget->setAbsCoor2D(150,200);
    this->updateElectronicWidget();
    if (this->hasPlayer())
      State::getPlayer()->hud().setEnergyWidget(this->electronicWidget);
  }
}

void SpaceShip::updateShieldWidget()
{
  if (this->shieldWidget != NULL)
  {
    this->shieldWidget->setMaximum(this->shieldMax);
    this->shieldWidget->setValue(this->shieldCur);;
  }
  else
  {
    this->shieldWidget = new OrxGui::GLGuiEnergyWidgetVertical();
    this->shieldWidget->getBarWidget()->setChangedValueColor(Color(1,0,0,1));
    //this->shieldWidget->setDisplayedName("Shield:");
    //his->shieldWidget->setSize2D(100,20);
    //this->shieldWidget->setAbsCoor2D(200,200);
    this->updateShieldWidget();
    if (this->hasPlayer())
      State::getPlayer()->hud().setShiledWidget(this->shieldWidget);
  }
}

void SpaceShip::setCameraDistance(float dist)
{
  Camera* c = State::getCamera();
  c->setViewTopDistance(dist);

  if (this->hasPlayer())
    this->isTravelDistanceInit = false;
}

void SpaceShip::setCameraFovy(float fovy)
{

  Camera* c = State::getCamera();
  c->setViewTopFovy(fovy);

  if (this->hasPlayer())
    this->isTravelDistanceInit = false;
}

void SpaceShip::updateTravelDistance()
{

  Camera* c = State::getCamera();

  float x = 1.25 * this->actionWidthPercentage * fabsf(c->getAbsCoor().y) * tan(c->getFovy()*M_PI /360.0);
  float y = x / c->getAspectRatio() / this->actionWidthPercentage;
  //State::getCamera()->setAbsCoor(-5, 1000, 0);


  //State::getCamera()->getAbsCoor().print();
  //printf("CameraRelCoorY: %f \n", State::getCamera()->getRelCoor().y);

  //printf("x: %f, y: %f \n", x, y);
  this->travelDistancePlus = Vector2D(y, x);
  this->travelDistanceMinus = Vector2D(-y, -x);

  State::getPlayer()->hud().setOverlayPercentage(100-int(100*this->actionWidthPercentage));
//   PRINTF(0)("TravelDistance has been updated\n");
  this->isTravelDistanceInit = true;
}

void SpaceShip::setActionWidthPercentage(int i)
{
  if (i>100) i=100;
  if (i<0) i=0;
  this->actionWidthPercentage = i/100.0;

  if (this->hasPlayer())
    this->isTravelDistanceInit = false;
};

void SpaceShip::addWeaponToSlot(int wm, int config, int slot, const std::string& weaponName){
  if (wm == 1) { this->weaponMan.addWeapon( Weapon::createWeapon( weaponName ), config, slot); }
  if (wm == 1) { this->weaponMan.addWeapon( Weapon::createWeapon( weaponName ), config, slot); }
  else return;
}

