/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:

   the actual playing shoult take place within this box

   main-programmer: Christoph Renner
*/

#include "action_box.h"
#include "track.h"
#include "state.h"

/**
 * this function is called each frame. update your state here
 * @param time time since last tick
 */
void ActionBox::tick( float time )
{
  updatePlanes();
}

/**
 * draws the box for debugging
 */
void ActionBox::draw( ) const
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  glPushAttrib(GL_ENABLE_BIT);

  glDisable(GL_LIGHTING);
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_BLEND);
  glLineWidth(2.0);
  glTranslatef ( track->getTrackNode()->getAbsCoor ().x,
                track->getTrackNode()->getAbsCoor ().y,
                track->getTrackNode()->getAbsCoor ().z);
  Vector tmpRot = track->getTrackNode()->getAbsDir().getSpacialAxis();
  glRotatef (track->getTrackNode()->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );


  glColor3f(1.0, 1.0, 0.0);
  glBegin(GL_LINE_STRIP);
    glVertex3f(0, height_2, width_2);
    glVertex3f(0, -height_2, width_2);
    glVertex3f(0, -height_2, -width_2);
    glVertex3f(0, height_2, -width_2);
    glVertex3f(0, height_2, width_2);
  glEnd();

  glColor3f(1.0, 0.0, 0.0 );
  glBegin(GL_LINE_STRIP);
    glVertex3f(depth, height_2 * stretch, width_2 * stretch);
    glVertex3f(depth, -height_2 * stretch, width_2 * stretch);
    glVertex3f(depth, -height_2 * stretch, -width_2 * stretch);
    glVertex3f(depth, height_2 * stretch, -width_2 * stretch);
    glVertex3f(depth, height_2 * stretch, width_2 * stretch);
  glEnd();

  glBegin(GL_LINE_STRIP);
    glVertex3f(depth, height_2 * stretch, width_2 * stretch);
    glVertex3f(0, height_2, width_2);
    glVertex3f(0, -height_2, width_2);
    glVertex3f(depth, -height_2 * stretch, width_2 * stretch);
  glEnd();

  glBegin(GL_LINE_STRIP);
    glVertex3f(depth, height_2 * stretch, -width_2 * stretch);
    glVertex3f(0, height_2, -width_2);
    glVertex3f(0, -height_2, -width_2);
    glVertex3f(depth, -height_2 * stretch, -width_2 * stretch);
  glEnd();

  glPopMatrix();
}

/**
 * create a new action box. you must not create more than one action box
 * @param _track track to assign actionbox to 
 * @param width_2 width/2 of near plane
 * @param height_2 height/2 of near plane
 * @param depth distance between near and far plane
 * @param stretch far plane will be stretched by this factor
 */
ActionBox::ActionBox( Track* _track, float width_2, float height_2, float depth, float stretch )
{
  assert( _track );
  assert( State::getActionBox() == NULL );
  
  State::setActionBox( this );
  
  this->width_2 = width_2;
  this->height_2 = height_2;
  this->depth = depth;
  this->stretch = stretch;
  this->track = _track;
  
  setParent( _track->getTrackNode() );
    
  toList( OM_COMMON );
}

/**
 * destructor
 */
ActionBox::~ActionBox( )
{
  assert( State::getActionBox() );
  State::setActionBox( NULL );
}

/**
 * checks wether a point is inside this box
 * @param pos point to check (absCoor)
 * @return true if inside
 */
bool ActionBox::isPointInBox( const Vector & pos )
{
  bool result = true;
  for ( int i = 0; i<6; i++ )
  {
    if ( planes[i].distancePoint( pos ) < 0 )
    {
      result = false;
      printf("PLANE %i FAILED %f\n", i, planes[i].distancePoint( pos ) );
    }
  }
  
  return result;
}

/**
 * update planes with new coordiantes. nomals must point to the inside
 */
void ActionBox::updatePlanes( )
{
  Vector p[7];
  
  p[0] = Vector( 0.0, height_2, -width_2 );
  p[1] = Vector( 0.0, height_2, width_2 );
  p[2] = Vector( 0.0, -height_2, -width_2 );
  p[3] = Vector( depth, height_2, -width_2 );
  p[4] = Vector( depth, height_2, width_2 );
  p[5] = Vector( depth, -height_2, width_2 );
  p[6] = Vector( depth, -height_2, -width_2 );
  
  for ( int i = 0; i<7; i++ )
  {
    p[i] = track->getTrackNode()->getAbsDir().apply( p[i] );
    p[i] += track->getTrackNode()->getAbsCoor();
  }
  
  planes[0] = Plane( p[1], p[0], p[2] );
  planes[1] = Plane( p[3], p[4], p[5] );
  planes[2] = Plane( p[5], p[4], p[1] );
  planes[3] = Plane( p[2], p[0], p[3] );
  planes[4] = Plane( p[3], p[0], p[1] );
  planes[5] = Plane( p[2], p[6], p[5] );
}



