/*
 *   ORXONOX - the hottest 3D action shooter ever to exist
 *                    > www.orxonox.net <
 *
 *
 *   License notice:
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation; either version 2
 *   of the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 *   Author:
 *      Fabien Vultier
 *   Co-authors:
 *      ...
 *
 */

/**
    @file IceGunProjectile.cc
    @brief Implementation of the IceGunProjectile class.
*/

#include "IceGunProjectile.h"

#include "core/CoreIncludes.h"
#include "graphics/Model.h"

namespace orxonox
{
    RegisterClass(IceGunProjectile);

    IceGunProjectile::IceGunProjectile(Context* context) : Projectile(context)
    {
        RegisterObject(IceGunProjectile);

        this->setCollisionShapeRadius(8.0f);

        this->setFreezeTime(3.0);
        this->setFreezeFactor(0.5);

        Model* model = new Model(this->getContext());
        model->setMeshSource("IceBolt.mesh");
        model->setScale(15.0);
        this->attach(model);
        model->setPosition(Vector3(0,0,0));
    }

    /**
    @brief
        Sets the freeze time variable to the passed value.
    */
    void IceGunProjectile::setFreezeTime(float freezeTime)
    {
        freezeTime_ = freezeTime;
    }

    /**
    @brief
        Sets the freeze factor variable to the passed value.
    */
    void IceGunProjectile::setFreezeFactor(float freezeFactor)
    {
        freezeFactor_ = freezeFactor;
    }

    bool IceGunProjectile::collidesAgainst(WorldEntity* otherObject, const btCollisionShape* cs, btManifoldPoint& contactPoint)
    {        
        bool bCollision = Projectile::collidesAgainst(otherObject, cs, contactPoint);

        if (bCollision)
        {
            // If there was a collision, attach a IceGunFreezer to the hit object.
            IceGunFreezer* freezer = new IceGunFreezer(this->getContext());
            freezer->setFreezeTime(freezeTime_);
            freezer->setFreezeFactor(freezeFactor_);
            otherObject->attach(freezer);            

            Vector3 offset = this->getWorldPosition() - otherObject->getWorldPosition();
            freezer->setPosition(Vector3(0,0,0));
            freezer->translate(offset, WorldEntity::World);
            // Start the freezing effect.
            freezer->startFreezing();
        }

        return bCollision;
    }
}
