/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Filip Gospodinov
   co-programmer: Silvan Nellen
*/

#include "cameraman.h"
#include "blackscreen.h"
#include "loading/load_param_xml.h"
#include "sound_engine.h"
#include "script_class.h"

ObjectListDefinition(CameraMan);

CREATE_SCRIPTABLE_CLASS(CameraMan,
			/// Camera management
                        addMethod("changeTarget", Executor3<CameraMan, lua_State*, const std::string&, const std::string&,const std::string&>(&CameraMan::changeTarget))
                        ->addMethod("attachCamera", Executor3<CameraMan, lua_State*,const std::string&,const std::string&,const std::string&>(&CameraMan::attachCamera))
			->addMethod("jumpCam", Executor4<CameraMan, lua_State*,const std::string&,float,float,float>(&CameraMan::jumpCam))
			->addMethod("moveCam", Executor4<CameraMan, lua_State*,const std::string&,float,float,float>(&CameraMan::moveCam))
                        ->addMethod("setRelCoor", Executor4<CameraMan, lua_State*,const std::string&,float,float,float>(&CameraMan::setRelCameraCoor))
                        ->addMethod("setRelCoorSoft", Executor5<CameraMan, lua_State*,const std::string&,float,float,float,float>(&CameraMan::setRelCameraCoorSoft))
			->addMethod("setCam", Executor1<CameraMan, lua_State*, const std::string&>(&CameraMan::setCam))
                        /// Current Camera
                        ->addMethod("changeCurrTarget", Executor2<CameraMan, lua_State*,const std::string&,const std::string&>(&CameraMan::changeCurrTarget))
                        ->addMethod("attachCurrCamera", Executor2<CameraMan, lua_State*,const std::string&,const std::string&>(&CameraMan::attachCurrCamera))
                        ->addMethod("jumpCurrCam", Executor3<CameraMan, lua_State*,float,float,float>(&CameraMan::jumpCurrCam))
                        ->addMethod("moveCurrCam", Executor3<CameraMan, lua_State*,float,float,float>(&CameraMan::moveCurrCam))
                        /// Fading
                        ->addMethod("toggleFade", Executor0<CameraMan, lua_State*>(&CameraMan::togglFade))
                        ->addMethod("initFadeBlack", Executor0<CameraMan, lua_State*>(&CameraMan::initFadeBlack))
			/// Polling
                        ->addMethod("getCurrCameraCoorX", Executor0ret<CameraMan, lua_State*,float>(&CameraMan::getCurrCameraCoorX))
                        ->addMethod("getCurrCameraCoorY", Executor0ret<CameraMan, lua_State*,float>(&CameraMan::getCurrCameraCoorY))
                        ->addMethod("getCurrCameraCoorZ", Executor0ret<CameraMan, lua_State*,float>(&CameraMan::getCurrCameraCoorZ))
			->addMethod("getCurrCamName", Executor0ret<CameraMan, lua_State*,const std::string&>(&CameraMan::getCurrCamName))
                       );



/**
 * Constructor of the CameraManager
 */
CameraMan::CameraMan(const TiXmlElement* root)
{
  this->registerObject(this, CameraMan::_objectList);

  this->nearClip = 1.0;
  this->farClip = 1000.0;

  this->fadeToBlack=new BlackScreen();

  this->setCam( State::getCamera() );

  if (root != NULL)
    this->loadParams(root);
}

/**
 *  Loads the parameters of the CameraManager from the xml file
 * @param root reference to the xml root element
 */
void CameraMan::loadParams(const TiXmlElement* root)
{
  BaseObject::loadParams(root);
  LoadParamXML(root, "Cameras", this, CameraMan, createCameras);
}

/**
 *  Creates all the Cameras
 * @param root reference to the xml root element
 */
void CameraMan::createCameras(const TiXmlElement* camerasTag)
{
    LOAD_PARAM_START_CYCLE(camerasTag, object);
    {
      this->createCam(object);
    }
    LOAD_PARAM_END_CYCLE(object);
}

/**
 *  Creates a Camera from a xml node
 * @param root reference to the xml node
 */
void CameraMan::createCam(const TiXmlElement* root)
{
  Camera* newCam = new Camera(root);
  newCam->setClipRegion(nearClip, farClip);
}

/**
 *  Changes the target of a camera. (where the camera looks at)
 *
 * @param cameraName the name of the camera whose target is changed
 * @param className the class of the new target
 * @param objectName the name of the new target
 */
void CameraMan::changeTarget(const std::string& cameraName,const std::string& className, const std::string& objectName)
{
  BaseObject* object = ObjectListBase::getBaseObject(className, objectName);
  Camera* camera = Camera::objectList().getObject(cameraName);

  if( object != NULL && camera != NULL && object->isA(PNode::staticClassID()))
  {
    camera->lookAt(dynamic_cast<PNode*>(object));
    return;
  }

 printf("ERROR CAMERAMANAGER: Couldn't change target for camera %s to: %s %s \n", cameraName.c_str(), className.c_str(),objectName.c_str() );

}

/**
 *  Attaches a camera to a new pnode. (the camera follows the pnode)
 *
 * @param cameraName the name of the camera
 * @param className the class of the new parent node
 * @param objectName the name of the new parent node
 */
void CameraMan::attachCamera(const std::string& cameraName, const std::string& className, const std::string& targetEntity)
{
  BaseObject* object = ObjectListBase::getBaseObject(className, targetEntity);
  Camera* camera = Camera::objectList().getObject(cameraName);

  if( object != NULL && camera != NULL && object->isA(PNode::staticClassID()) )
  {
    camera->target->atach(dynamic_cast<PNode*>(object));
    camera->setViewMode(Camera::ViewNormal);
    return;
  }

  printf("ERROR CAMERAMANAGER: Couldn't attach camera %s to: %s %s \n", cameraName.c_str(), className.c_str(),targetEntity.c_str() );

}


/**
 *  Sets a camera to a new position
 *
 * @param cameraName the name of the camera
 * @param x the new x coordinate
 * @param y the new y coordinate
 * @param z the new z coordinate
 */
void CameraMan::jumpCam(const std::string& cameraName, float x, float y, float z)
{
  Camera* camera = Camera::objectList().getObject(cameraName);
  if( camera != NULL )
  {
    camera->target->jump( x, y, z );
  }
}

/**
 *  Moves a camera slowly to a new position
 *
 * @param cameraName the name of the camera
 * @param x the new x coordinate
 * @param y the new y coordinate
 * @param z the new z coordinate
 */
void CameraMan::moveCam(const std::string& cameraName, float x, float y, float z)
{
  Camera* camera = Camera::objectList().getObject(cameraName);
  if( camera != NULL )
  {
    camera->target->trans( x, y, z );
  }
}

/**
 *  Sets the coordinate of a camera relative to its target.
 *
 * @param cameraName the name of the camera
 * @param x the new x coordinate
 * @param y the new y coordinate
 * @param z the new z coordinate
 */
void CameraMan::setRelCameraCoor(const std::string& cameraName, float x, float y, float z)
{
  Camera* camera = Camera::objectList().getObject(cameraName);
  camera->setRelCoor(x,y,z);
  camera->target->setRelCoor(0,0,0);
}

/**
 *  Sets the coordinate of a camera relative to its target. (softly) 
 *
 * @param cameraName the name of the camera
 * @param x the new x coordinate
 * @param y the new y coordinate
 * @param z the new z coordinate
 */
void CameraMan::setRelCameraCoorSoft(const std::string& cameraName, float x, float y, float z, float bias)
{
  Camera* newCam = Camera::objectList().getObject(cameraName);
  newCam->setRelCoor(x,y,z);
  newCam->target->setRelCoorSoft(0,0,0);
}

/**
 *  Sets a new camera as the current one.
 *
 * @param cameraName the name of the new camera
 */
void CameraMan::setCam(const std::string& cameraName)
{
  Camera* camera = Camera::objectList().getObject(cameraName);

  if(camera != NULL)
  {
    this->setCam(camera);
    return;
  }
  printf("ERROR CAMERAMANAGER: Couldn't set camera : %s \n", cameraName.c_str());
}

/**
 *  Sets a new camera as the current one.
 *
 * @param camera a pointer to the new camera
 */
void CameraMan::setCam(Camera* newCamera)
{
  if( newCamera == NULL)
  {
    PRINTF(0)("trying to add a zero camera! uiuiui!\n");
    return;
  }
  
  State::setCamera(newCamera, newCamera->getTarget());
  OrxSound::SoundEngine::getInstance()->setListener(newCamera);

  this->fadeToBlack->setRelCoor(0., 0., 0.);
  this->fadeToBlack->setParent(newCamera);

}

/**
 *  Sets the clipregion of all the cameras
 *
 * @param nearClip the new near clip
 * @param nearClip the new far clip
 */
void CameraMan::setClipRegion(float nearClip, float farClip)
{
  this->nearClip=nearClip;
  this->farClip=farClip;

  for (ObjectList<Camera>::const_iterator it = Camera::objectList().begin();
       it != Camera::objectList().end();
       ++it)
  {
     (*it)->setClipRegion(this->nearClip, this->farClip);
  }
}

/**
 *  Changes the target of the current camera. (where the camera looks at)
 *
 * @param className the class of the new target
 * @param objectName the name of the new target
 */
void CameraMan::changeCurrTarget(const std::string& className, const std::string& objectName)
{
  this->changeTarget( (State::getCamera())->getName() , className, objectName);
}

/**
 *  Attaches the current camera to a new pnode. (the camera follows the pnode)
 *
 * @param className the class of the new parent node
 * @param objectName the name of the new parent node
 */
void CameraMan::attachCurrCamera(const std::string& className, const std::string& targetEntity)
{
  this->attachCamera( (State::getCamera())->getName(), className,  targetEntity);
}

/**
 *  Sets the current camera to a new position
 *
 * @param x the new x coordinate
 * @param y the new y coordinate
 * @param z the new z coordinate
 */
void CameraMan::jumpCurrCam(float x, float y, float z)
{
  this->jumpCam((State::getCamera())->getName(), x, y, z);
}

/**
 *  Moves the current camera slowly to a new position
 *
 * @param x the new x coordinate
 * @param y the new y coordinate
 * @param z the new z coordinate
 */
void CameraMan::moveCurrCam(float x, float y, float z)
{
  this->moveCam( (State::getCamera())->getName(), x, y, z);
}

/**
 *  Toggles the fadestate (from in to out and via versa)
 */
void CameraMan::togglFade()
{
 if( this->fadeToBlack)
    fadeToBlack->toggleFade();
}

/**
 *  Sets the fadestate immediately to black (fade out)
 */
void CameraMan::initFadeBlack()
{
  if( this->fadeToBlack)
    fadeToBlack->initFadeBlack();
}
