/*!
 * @file spawning_point.h
 *  Definition of a spawning point within the game, used for network game
 */


#ifndef _SPAWNING_POINT
#define _SPAWNING_POINT

#include "playable.h"

#include "message_manager.h"

#include <list>

class World;
class TiXmlElement;

struct QueueEntry
{
  float respawnTime;
  Playable * entity;
};

//!< used to indicate what type of objects are spawned by this spawning point
typedef enum SpawningPointMode
{
  SPT_ALL_AT_ONCE = 0,                               //!< at this spawning points there will be players spawned (typicaly in MP games)
  SPT_ONE_AFTER_OTHER,                               //!< at this spawning points there will be NPS spawnded

  SPT_NUMBER
};


/**
 * The spawning point for WorldEntities (and only WorldEntities)
 *
 * There are commonly two different spawning modes:
 *
 *  1) Just spawn whatever is in the queue with a given frequency (if delay = 0 => immediate spawn)
 *  2) Spawn everything in the queue together with the given frequency
 */
class SpawningPoint : public WorldEntity {
  ObjectListDeclaration(SpawningPoint);
  public:
    SpawningPoint(const TiXmlElement* root = NULL);
    virtual ~SpawningPoint ();
    void init();

    virtual void loadParams(const TiXmlElement* root);

    inline int getTeamId() const { return this->teamId; }
    inline void setTeamId( int teamId ) { this->teamId = teamId; }

    void pushEntity(Playable* entity, float delay = 0);

    /** activates the spawning point */
    inline void activate() { this->bSpawning = true; }
    /** deactivates the spawning point */
    inline void deactivate() { this->bSpawning = false; }
    inline bool isActive() const { return this->bSpawning; }


    virtual void tick(float dt);
    virtual void draw() const;


  private:
    void spawn(Playable* entity);

    void sendRespawnMessage( int uniqueId );
    static bool respawnMessageHandler( MessageType messageType, byte * data, int dataLength, void * someData, int senderId, int destinationId  );


  private:
    float                           localTimer;                     //!< the local timer
    int                             teamId;                         //!< only spawn players of this team
    std::list<QueueEntry>           queue;                          //!< queue of waiting WorldEntities to be spawned
    bool                            bSpawning;                      //!< flag to indicate if this spawning point is active or not
};

#endif /* _SPAWNING_POINT */
