/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Filip Gospodinov
   co-programmer:
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "scrolling_screen.h"

#include "util/loading/factory.h"
#include "util/loading/load_param.h"

#include "debug.h"
#include "material.h"
#include "state.h"
#include "cameraman.h"
#include "camera.h"

#include "script_class.h"


ObjectListDefinition(ScrollingScreen);
CREATE_FACTORY(ScrollingScreen);

CREATE_SCRIPTABLE_CLASS(ScrollingScreen,
                        addMethod("start", Executor0<ScrollingScreen, lua_State*>(&ScrollingScreen::start))
                       );


/**
 *
 */
ScrollingScreen::ScrollingScreen()
{
  this->init();
}


/**
 *
 */
ScrollingScreen::ScrollingScreen(const TiXmlElement* root)
{
  this->init();

  if( root != NULL)
    this->loadParams(root);
}


/**
 *
 */
ScrollingScreen::~ScrollingScreen()
{}

void ScrollingScreen::start()
{
  this->bRun = true;
}

/**
 *
 */
void ScrollingScreen::init()
{
  this->registerObject(this, ScrollingScreen::_objectList);
  this->toList(OM_COMMON);

  this->material = new Material();
  this->material->setDiffuse(1,1,1);
  this->material->setBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  this->isTransparent = false;
  this->transparency = 1.0;
  this->offset = 0.0;

  this->xSize = 0.;
  this->ySize = 0.;
  this->mode = SCSC_VIEW;
  this->bRun = false;
}


/**
 * loads the Settings of a MD2Creature from an XML-element.
 * @param root the XML-element to load the MD2Creature's properties from
 */
void ScrollingScreen::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "setSpeed", this, ScrollingScreen, setSpeed);

  LoadParam(root, "setHeight", this, ScrollingScreen, setViewHeight);

  LoadParam(root, "setSizeX", this, ScrollingScreen, setSizeX);
  LoadParam(root, "setSizeY", this, ScrollingScreen, setSizeY);

  LoadParam(root, "texture", this, ScrollingScreen, setTexture);
}




/**
 * sets the texture
 * @param texture name of tex
 */
void ScrollingScreen::setTexture(const std::string& texture)
{
  this->material->setDiffuseMap(texture);

  Texture t = this->material->diffuseTexture();
  this->ratio = t.getWidth() / t.getHeight();
}




void ScrollingScreen::draw() const
{

  if( this->offset - this->ratio > 1.|| this->offset < -1. || !this->bRun)
    return;

  if( this->mode == SCSC_FULL)
    this->drawFull();
  else
    this->drawView();
}

void ScrollingScreen::drawFull() const
{
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  Vector tmpRot = this->getAbsDir().getSpacialAxis();
  glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );

  this->material->select();


  glBegin(GL_QUADS);

  // unten links
  glTexCoord2f(0., 1.);
  glVertex3f(0., -this->xSize*0.5, -this->ySize*0.5);

  // unten rechts
  glTexCoord2f(1., 1.);
  glVertex3f(0., -this->xSize*0.5, this->ySize*0.5);

  // oben rechts
  glTexCoord2f(1., 0.);
  glVertex3f(0., this->xSize*0.5, this->ySize*0.5);

  // oben links
  glTexCoord2f(0., 0.);
  glVertex3f(0., this->xSize*0.5, -this->ySize*0.5);

  glEnd();

  glPopAttrib();
  glPopMatrix();
}

void ScrollingScreen::drawView() const
{
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  Vector tmpRot = this->getAbsDir().getSpacialAxis();
  glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );

  this->material->select();

  float resize = (1. - (this->viewHeight + this->offset));

  float x,y;
  if( this->xSize != 0.)
  {
    x = this->xSize;
    y = x * this->ratio;
  }
  else
  {
    y = this->ySize;
    x = y * this->ratio;
  }

  float ratShift = 0.;
  float borderTex = this->offset + ratio;
  // we are above/below
  if( 1 - (offset + ratio) < 0)
  {
    borderTex = 1.0;
    ratShift = 1. - (1.-offset)/ratio;
  }

  glBegin(GL_QUADS);
  // unten links
  glTexCoord2f(0., borderTex);
  glVertex3f(0., -(x*0.5 - x*ratShift), -y*0.5);

  // unten rechts
  glTexCoord2f(1., borderTex);
  glVertex3f(0., -(x*0.5 - x*ratShift), y*0.5);

  // oben rechts
  glTexCoord2f(1., this->offset);
  glVertex3f(0., x*0.5, y*0.5);

  // oben links
  glTexCoord2f(0., this->offset);
  glVertex3f(0., x*0.5, -y*0.5);

  glEnd();

  glPopAttrib();
  glPopMatrix();
}


/**
 *
 */
void ScrollingScreen::tick (float time)
{

  if( !this->bRun)
    return;

  CameraMan* cm = State::getCameraman();
  const Camera* cam = cm->getCurrentCam();
  PNode* tar = cam->getTargetNode();

  Vector dir = tar->getAbsCoor() - cam->getAbsCoor();
  dir = dir.getNormalized();

  float offset = 6.;

  this->setAbsCoor( cam->getAbsCoor() + dir * offset);


  Vector ddir = dir.cross( cam->getAbsDirV());
  Quaternion q(ddir, cam->getAbsDirY());
  this->setAbsDir( q);

  // scroll the texture
  this->offset += time * this->scrollingSpeed;
  //     if( this->offset - this->ratio > 1.|| this->offset < -1.)
  //       this->offset = 0.;

}

void ScrollingScreen::fadeIn(float speed)
{
}

void ScrollingScreen::fadeOut(float speed)
{
}
