/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2005 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: David Hasenfratz, Stefan Lienhard
   co-programmer:
*/

#include "movie_entity.h"

#include "media_container.h"
#include "util/loading/load_param.h"
#include "util/loading/factory.h"



ObjectListDefinition(MovieEntity);
CREATE_FACTORY(MovieEntity);

/**
 *  standard constructor
 */
MovieEntity::MovieEntity (const TiXmlElement* root)
{
  this->registerObject(this, MovieEntity::_objectList);

  media_container = new MediaContainer();

  axis = 0;
  rotation = 0;
  height = 20;
  width = 20;
  mediaLoaded = false;

  this->toList(OM_COMMON);

  if( root != NULL)
    this->loadParams(root);

  counter = 0;
  timer = 0;
}

/**
 *  standard destructor
 */
MovieEntity::~MovieEntity ()
{
  if( this->media_container)
    delete this->media_container;
}

void MovieEntity::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "name", this, MovieEntity, loadMovie);
  LoadParam(root, "axis", this, MovieEntity, setAxis);
  LoadParam(root, "rotation", this, MovieEntity, setRotation);
  LoadParam(root, "size", this, MovieEntity, setSize);
  LoadParam(root, "fps", this, MovieEntity, setFPS);
}

void MovieEntity::setFPS(float fps)
{
  this->fps = fps;
//PRINTF(0)("fps: %f\n", fps);
}

void MovieEntity::loadMovie(const std::string& filename)
{
  if(media_container->loadMedia(filename))
  {
    mediaLoaded = true;
    fps = media_container->getFPS();
  }
  else
    mediaLoaded = false;
}

void MovieEntity::setAxis(float axis)
{
//PRINTF(0)("fps: %f\n", fps);
  this->axis = axis;
}

// Seconds for one loop
void MovieEntity::setRotation(float rotation)
{
  this->rotation = rotation;
}

void MovieEntity::setSize(float width, float height)
{
  this->width = width;
  this->height = height;
}

/**
 *  this method is called every frame
 * @param time: the time in seconds that has passed since the last tick

   Handle all stuff that should update with time inside this method (movement, animation, etc.)
 */
void MovieEntity::tick(float time)
{
  if(!mediaLoaded)
    return;

  timer += time;

  if(counter != fps * timer)
  {
    counter = (int)(fps * timer);

    if ((unsigned int)counter >= media_container->getFrameCount())
    {
      timer = 0;
      counter = 0;
    }
  }

  axis += (time * rotation/360.0);
}


/**
 *  the entity is drawn onto the screen with this function

   This is a central function of an entity: call it to let the entity painted to the screen. Just override this function with whatever you want to be drawn.
 */
void MovieEntity::draw() const
{
//   if(!mediaLoaded)
//     false;

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  glDisable(GL_BLEND);

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, media_container->getFrameTexture(counter));

  glPushMatrix();
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  glRotatef(axis, 0.0f, 1.0f, 0.0f);
//PRINTF(0)("axis: %f\n", axis);

  glColor3f(1.0, 1.0, 1.0);

  glBegin(GL_QUADS);
    glTexCoord2f(1.0f, 1.0f); glVertex3f(-width/2, -height/2,  0.0f);
    glTexCoord2f(0.0f, 1.0f); glVertex3f( width/2, -height/2,  0.0f);
    glTexCoord2f(0.0f, 0.0f); glVertex3f( width/2,  height/2,  0.0f);
    glTexCoord2f(1.0f, 0.0f); glVertex3f(-width/2,  height/2,  0.0f);
  glEnd();

  glPopMatrix();
  glPopAttrib();
}
