/*!
 * @file camera.h
 * Viewpoint controlling class definitions
*/

#ifndef _CAMERA_H
#define _CAMERA_H

#include "world_entity.h"
#include "event_listener.h"
#include "plane.h"


class World;
class CameraTarget;
class Event;

//! Camera
/**
 * This class controls the viewpoint from which the World is rendered.
*/
class Camera : public WorldEntity, public EventListener
{
  friend class CameraTarget;
  friend class CameraMan;
  ObjectListDeclaration(Camera);
public:
  //! an enumerator for different types of view
  typedef enum ViewMode
  {
    ViewNormal,
    ViewBehind,
    ViewFront,
    ViewLeft,
    ViewRight,
    ViewTop
  };
  public:
  Camera();
  Camera(const TiXmlElement* root);
  virtual ~Camera();

  void lookAt(PNode* target);
  CameraTarget* getTarget() const { return this->target; };
  PNode* getTargetNode() const;
  void setTargetNode(PNode* target);
  void setAspectRatio(float aspectRatio);
  inline float getAspectRatio() {return this->aspectRatio;};

  void setClipRegion(float nearClip, float farClip);

  /** @param fovy new field of view factor (in degrees) */
  inline void setFovy(float fovy)
  {
    this->fovy = fovy;
    this->toFovy = fovy;
  };

  inline float getFovy() {return this->fovy;};
  /** @param fovy new field of view factor (in degrees) to iterate to */
  void setToFovy(float toFovy) { this->toFovy = toFovy; };

  void setViewMode(Camera::ViewMode mode);
  inline const Vector& getViewVector() const { return this->viewVector; }
  inline const Vector& getUpVector() const { return this->upVector; }
  inline const Plane& getViewFrustum() const { return this->frustumPlane; }

  inline float distance(const Vector& distance) const { return this->frustumPlane.distancePoint(distance); }
  inline float distance(const PNode* node) const { return distance(node->getAbsCoor()); }

  inline void setEventHandling(bool b) {this->eventHandling = b;}
  inline bool getEventHandling() {return this->eventHandling;}

  void glLookAt(float eyex, float eyey, float eyez, float centerx, float centery, float centerz, float upx, float upy, float upz);
  Vector* VectorProd(Vector* v1, Vector* v2);
  void Rotate();
  void draw() const;
  void tick(float dt);
  void apply ();
  void project();

  void process(const Event &event);
  //CameraTarget*     target;          //!< The Target of the Camera (where this Camera Looks at)

  virtual void loadParams(const TiXmlElement* root);

  void              setViewTopFovy(float fovy);
  void              setViewLeftFovy(float fovy);
  void              setViewRightFovy(float fovy);
  void              setViewBehindFovy(float fovy);
  void              setViewFrontFovy(float fovy);
  void              setViewNormalFovy(float fovy);

  void              setViewTopDistance(float Distance);
  void              setViewLeftDistance(float Distance);
  void              setViewRightDistance(float Distance);
  void              setViewBehindDistance(float Distance);
  void              setViewFrontDistance(float Distance);
  void              setViewNormalDistance(float Distance);

private:

  void              init();

  CameraTarget*     target;          //!< The Target of the Camera (where this Camera Looks at)

  bool              eventHandling;    //!< True, if the Camera handles the processing of events itself. Set false to overwrite the standard handling.

  float             fovy;            //!< The field of view Angle (in degrees).
  float             aspectRatio;     //!< The aspect ratio (width / height).
  float             nearClip;        //!< The near clipping plane.
  float             farClip;         //!< The far clipping plane.

  float             toFovy;          //!< The fovy-mode to iterate to.
  Camera::ViewMode  currentMode;     //!< The ViewMode the camera is in

  Vector            delay;
  Plane             frustumPlane;    //!< plane that marks the view frustum
  Vector            viewVector;      //!< the direction of the camera view
  Vector            upVector;        //!< direction of the up vector

  float             viewTopFovy;
  float             viewLeftFovy;
  float             viewRightFovy;
  float             viewBehindFovy;
  float             viewFrontFovy;
  float             viewNormalFovy;

  float             viewTopDistance;
  float             viewLeftDistance;
  float             viewRightDistance;
  float             viewBehindDistance;
  float             viewFrontDistance;
  float             viewNormalDistance;

};

//! A CameraTarget is where the Camera is looking at.
class CameraTarget : public PNode
{
  friend class Camera;        //! The CameraTarget is a friend of Camera. noone else needs a CameraTarget, so noone else can create it.
  ObjectListDeclaration(CameraTarget);

private:
  CameraTarget();
  virtual ~CameraTarget() {}
  float speed;
  PNode* target;
  PNode* freeTarget;
  Camera* masta;
  Vector translateTo;
  Vector rotateBy;


public:

  void detach();
  void atach(PNode* object);
  Vector iterate(float dt, const Vector* target, const Vector* cam);
  void translate(float dt);
  void changeSpeed(float speed);
  Vector* rotate(Vector* newPos, float speed);
  void jump(float x, float y, float z);
  void translateNow(Vector* vec);
  PNode* createStick();
  void trans(float x, float y, float z);
  bool isDone();
};




#endif /* _CAMERA_H */

