/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Manuel Leuenberger
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "power_up.h"
#include "extendable.h"
#include "primitive_model.h"

#include "sound/resource_sound_buffer.h"
#include "util/loading/load_param.h"

#include "debug.h"

ObjectListDefinition(PowerUp);

PowerUp::PowerUp(float r, float g, float b)
{
  this->registerObject(this, PowerUp::_objectList);

  this->respawnType = RESPAWN_TIME;
  this->respawnStart = 10;
  this->model = NULL;
  /*  if(!PowerUp::sphereModel) {*/

  Model* sphereModel = new PrimitiveModel(PRIM_SPHERE, 7, 5);

  this->setModel(sphereModel);
  this->buildObbTree( 4);
  this->sphereMaterial = new Material;
  this->sphereMaterial->setTransparency(.8);
  this->sphereMaterial->setDiffuse(r, g, b);
  this->toList(OM_COMMON);

  this->soundSource.setSourceNode(this);

  this->collider = NULL;
}

PowerUp::~PowerUp ()
{
  delete this->sphereMaterial;
}


void PowerUp::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "respawnType", this, PowerUp, setRespawnType);

  LoadParam(root, "respawnTime", this, PowerUp, setRespawnTime);

  LoadParam(root, "pickup-sound", this, PowerUp, loadPickupSound);

  LoadParam(root, "respawn-sound", this, PowerUp, loadRespawnSound);
}


void PowerUp::loadPickupSound(const std::string& pickupSound)
{
  if (!pickupSound.empty())
    this->pickupBuffer = OrxSound::ResourceSoundBuffer(pickupSound);
  else
    this->pickupBuffer = OrxSound::SoundBuffer();
}

void PowerUp::loadRespawnSound(const std::string& respawnSound)
{
  if (!respawnSound.empty())
    this->respawnBuffer = OrxSound::ResourceSoundBuffer(respawnSound);
  else
    this->respawnBuffer = OrxSound::SoundBuffer();
}


void PowerUp::collidesWith (WorldEntity* entity, const Vector& location)
{
  if(this->collider != entity && entity->isA(Extendable::staticClassID()))
  {
    this->collider = entity;
    if(dynamic_cast<Extendable*>(entity)->pickup(this))
    {
      if(pickupBuffer.loaded())
        this->soundSource.play(this->pickupBuffer);

      switch(respawnType)
      {
      case RESPAWN_NONE:
        this->toList(OM_DEAD);
        break;
      case RESPAWN_TIME:
        this->toList(OM_DEAD_TICK);
        this->respawnTime = this->respawnStart;
        break;
        default:
          /* NOT HANDLED */
          break;
      }
    }
  }
}

void PowerUp::tick(float dt)
{
  if(this->getOMListNumber() != OM_COMMON)
  {
    this->respawnTime -= dt;
    if(this->respawnTime <= 0)
    {
      this->toList(OM_COMMON);
      this->collider = NULL;
      if (likely(this->respawnBuffer.loaded()))
        this->soundSource.play(this->respawnBuffer);

    }
  }
}

void PowerUp::draw() const
{
  if(this->model != NULL)
  {
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef (this->getAbsCoor ().x,
                  this->getAbsCoor ().y,
                  this->getAbsCoor ().z);
    Vector tmpRot = this->getAbsDir().getSpacialAxis();
    glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
    this->model->draw();
    glPopMatrix();
  }
  this->sphereMaterial->select();
  WorldEntity::draw();
}

const char* PowerUp::respawnTypes[] =
  {
    "none",
    "time"
  };


void PowerUp::setRespawnType(const std::string& type)
{
  for(int i = 0; i < RESPAWN_size; ++i)
  {
    if(type == respawnTypes[i])
    {
      this->respawnType = (PowerUpRespawn)i;
      break;
    }
  }
}

void PowerUp::setRespawnTime(const float respawnTime)
{
  this->respawnStart = respawnTime;
}


