/*!
 * @file skybox.h
 *  Definition of the SkyBox, that handles the Display of an atmosphere for orxonox.
 *
 * A SkyBox is always centered at the current working Camera, and will only obey the cameras
 * movment but not its rotation.
*/

#ifndef _SKYBOX_H
#define _SKYBOX_H

/* INCLUDES */
#include "world_entity.h"
#include "material.h"

enum SKY_SIDE
{
  SKY_BACK = 0,
  SKY_FRONT,
  SKY_BOTTOM,
  SKY_TOP,
  SKY_LEFT,
  SKY_RIGHT,
};

//! A Class to handle a SkyBox
class SkyBox : public WorldEntity
{
  ObjectListDeclaration(SkyBox);
public:
  SkyBox(const std::string& fileName = "");
  SkyBox(const TiXmlElement* root);

  virtual ~SkyBox();

  void init();
  void preInit();

  virtual void loadParams(const TiXmlElement* root);

  void postInit();

  virtual void draw();

  void setSize(float size);
  /** assumes jpg as input-format */
  void setTexture(const std::string& name);

  void setTextureAndType(const std::string& name, const std::string& extension);
  void setTextures(const std::string& negX, const std::string& posX, const std::string& negY,
                   const std::string& posY, const std::string& negZ, const std::string& posZ);

  void loadCubeMapTextures(const std::string& negX, const std::string& posX, const std::string& negY,
                           const std::string& posY, const std::string& negZ, const std::string& posZ);

  GLuint getTexture(SKY_SIDE side) const { return (this->material[side]) ? this->material[side]->diffuseTextureID(): 0; };

  static void enableCubeMap();
  static void disableCubeMap();

  virtual void varChangeHandler( std::list<int> & id );

 private:
  void rebuild();

  Material*       material[6];     //!< Materials for the SkyBox. sorted by number (0-5) top, bottom, left, right, front, back
  Texture         cubeTexture[6];  //!< Textures for the CubeMap.

  float           size;            //!< Size of the SkyBox. This should match the frustum maximum range.
  float           textureSize;     //!< this is the length of a texture (assumes a square texture)
  std::string     textureName;     //!< Name of the Texture

  int textureName_handle;          //!< used to notify about changes of textureName
  int size_handle;                 //!< used to notify about changes of size

};

#endif  /* _SKYBOX_H */



