/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: David Hasenfratz
*/

#include "wobblegrid.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "graphics_engine.h"
#include "material.h"
#include "glincl.h"
#include "state.h"
#include "grid.h"

#include "tools/cameraman.h"
#include "tools/camera.h"

#include <cassert>
#include "debug.h"



ObjectListDefinition(Wobblegrid);
CREATE_FACTORY(Wobblegrid);

/**
 * Wobble Grids are grids which "wobble"
 * Wobbling is realized through fixed center and sin wave outwards
 * For the beginning the grid will be a 5x5
 */

/**
 * standart constructor
 */
Wobblegrid::Wobblegrid (const TiXmlElement* root)
{
  this->size  = 5;

  this->init();

  if( root)
    this->loadParams(root);
}

Wobblegrid::Wobblegrid (int size, const TiXmlElement* root)
{
  this->size  = size;

  this->init();

  if( root)
    this->loadParams(root);
}


/**
 * destroys a Wobblegrid
 */
Wobblegrid::~Wobblegrid ()
{
  if (this->material)
    delete this->material;
}


/**
 * initializes the Wobblegrid
 */
void Wobblegrid::init()
{
  this->registerObject(this, Wobblegrid::_objectList);
  this->setName("Wobblegrid");

  this->material = new Material();
  this->material->setBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  this->setAbsCoor(0, 0, 0);
  //this->setVisibiliy(true);
  
  this->subdivision = 5;

  this->grid  = new Grid( this->size, this->size, this->subdivision, this->subdivision);
  this->angularSpeed = M_PI; //180;
  this->setModel(this->grid);

  this->setUpdateFunction((*sinf));
}


/**
 *  load params
 * @param root TiXmlElement object
 */
void Wobblegrid::loadParams(const TiXmlElement* root)
{
  /*LoadParam(root, "texture", this->material, Material, setDiffuseMap)
      .describe("the texture-file to load onto the Wobblegrid");

  LoadParam(root, "size", this, Wobblegrid, setSize)
  .describe("the size of the Wobblegrid in Pixels");*/
}



/**
 * sets the material to load
 * @param textureFile The texture-file to load
 */
void Wobblegrid::setTexture(const std::string& textureFile)
{
  this->material->setBlendFunc(GL_SRC_ALPHA,GL_ONE);
  this->material->setDiffuseMap(textureFile);
}


/**
 * ticks the Wobblegrid
 * @param dt the time to ticks
 */
void Wobblegrid::tick(float dt)
{
  angle += dt * angularSpeed;
  if (angle > 2 * M_PI)
    angle -= 2 * M_PI;

  //Vector vec;
  float fac = 1.0 / (this->subdivision - 1);
  for( int z=1; z < 4; z++)
  {
    for( int x=1; x < 4; x++)
    {
      //if(x==2 && z == 2)
      //  continue;

      Vector2D& vec = this->grid->texCoord(z*this->subdivision + x);
      vec.y  = (x * fac + sgn(x-2)*updateWobble(angle)*fac/2.0);
      vec.x  = (z * fac + sgn(z-2)*updateWobble(angle)*fac/2.0);
    }
  }
  //this->grid->finalize();
  this->orient();
}


/**
 * draws the billboard
 */
void Wobblegrid::draw() const
{

//   this->material->select();
//   WorldEntity::draw();
//   return;


  if( !this->isVisible())
    return;

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  //glTranslatef(this->getAbsCoor().x, this->getAbsCoor().y, this->getAbsCoor().z);
  //glTranslatef(0,0,0);
  this->material->select();

  glDisable(GL_BLEND);
  glEnable( GL_ALPHA_TEST);
  glAlphaFunc( GL_GEQUAL, .5);

  glTranslatef (this->getAbsCoor ().x,
                  this->getAbsCoor ().y,
                  this->getAbsCoor ().z);

  Vector tmpRot = this->getAbsDir().getSpacialAxis();
  glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );

  //Quaternion  dir = Quaternion(this->getAbsDir().getSpacialAxisAngle(),view);
  //this->setAbsDir(dir);
  //glRotatef(this->angle, 0.0, 0.0, 1.0);
  this->grid->draw();

  glPopMatrix();

  glPopAttrib();
}

void Wobblegrid::orient()
{
     
     CameraMan* cman = State::getCameraman();
     Camera* c = cman->getCurrentCam();

  Vector view = this->getAbsCoor() - c->getAbsCoor();
//   view.normalize();

  Vector up = Vector(0, 1, 0);
//   if ( up.dot(view) == 0 )
//     up = Vector(1, 0, 0);
//   if ( up.dot(view) == 0 )
//     up = Vector (0, 0, 1);

  Vector h = up.cross(view.getNormalized());
  up = h.cross(view.getNormalized());
  Quaternion dir = Quaternion::lookAt( this->getAbsCoor(), this->getAbsCoor() + up, view);
//   Quaternion dir = Quaternion::lookAt( Vector(), view, up);
  this->setAbsDir(dir);
}
