/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004-2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Marc Schaerrer, Nicolas Schlumberger
   co-programmer:

*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WEAPON

#include "swarm_projectile.h"

#include "state.h"

#include "particles/dot_emitter.h"
#include "particles/sprite_particles.h"
#include "space_ships/space_ship.h"
#include "effects/trail.h"

#include "debug.h"



#include "math/vector.h"

ObjectListDefinition(SwarmProjectile);
CREATE_FAST_FACTORY_STATIC(SwarmProjectile);

/**
 *  standard constructor
*/
SwarmProjectile::SwarmProjectile () : Projectile()
{

/*  this->loadModel("models/projectiles/orx-rocket.obj", 0.5);*/
  this->loadModel("models/projectiles/swarm_projectile.obj", .25); // no double rescale (see draw())
  this->loadExplosionSound("sounds/explosions/explosion_4.wav");


  this->setMinEnergy(1);
  this->setHealthMax(10);
  this->lifeSpan = 4.0;
  this->agility = 3.5;

  this->emitter = new DotEmitter(100, 5, M_2_PI);
  this->emitter->setParent(this);
  this->emitter->setSpread(M_PI, M_PI);

  this->turningSpeed = 10;

  this->physDamage = 200;
  this->elecDamage = 0;

  this->trail = new Trail(2.5,4,.2, this);
  //this->trail->setParent( this);
  this->trail->setTexture( "textures/laser.png");


  this->origList = this->getOMListNumber();
  this->toList(OM_ENVIRON);
}


/**
 *  standard deconstructor
*/
SwarmProjectile::~SwarmProjectile ()
{

  if (SwarmProjectile::explosionParticles != NULL && SwarmProjectile::objectList().size() <= 1)
  {
    if (ParticleSystem::objectList().exists(SwarmProjectile::explosionParticles))
      delete SwarmProjectile::explosionParticles;
    SwarmProjectile::explosionParticles = NULL;
  }
  // delete this->emitter;
  delete this->trail;
}

SpriteParticles* SwarmProjectile::explosionParticles = NULL;



void SwarmProjectile::activate()
{
  this->toList(OM_ENVIRON);
  if (unlikely(SwarmProjectile::explosionParticles == NULL))
  {
    SwarmProjectile::explosionParticles = new SpriteParticles(200);
    SwarmProjectile::explosionParticles->setName("SwarmProjectileExplosionParticles");
    SwarmProjectile::explosionParticles->setMaterialTexture("textures/radial-trans-noise.png");
    SwarmProjectile::explosionParticles->setLifeSpan(.5, .3);
    SwarmProjectile::explosionParticles->setRadius(0.0, 10);
    SwarmProjectile::explosionParticles->setRadius(.5, 15.0);
    SwarmProjectile::explosionParticles->setRadius(1.0, 10.0);
    SwarmProjectile::explosionParticles->setColor(0.0, 0,1,0,1);
    SwarmProjectile::explosionParticles->setColor(0.5, .8,.8,0,.8);
    SwarmProjectile::explosionParticles->setColor(0.8, .8,.8,.3,.8);
    SwarmProjectile::explosionParticles->setColor(1.0, 1,1,1,.0);
  }

  this->emitter->setEmissionRate(50.0);
  this->emitter->setEmissionVelocity(0.0);
  this->emitter->setInheritSpeed(0);

  this->setHealth(10.0* (float)rand()/(float)RAND_MAX);

  this->maxVelocity = 300;

  this->rotationSpeed = 360;
  this->angle = 0;

  this->curDir = this->lastDir = this->velocity;
}


void SwarmProjectile::deactivate()
{
  this->emitter->setSystem(NULL);
  this->lifeCycle = 0.0;

  this->toList(OM_DEAD);
  this->removeNode();
  SwarmProjectile::fastFactory->kill(this);
}


void SwarmProjectile::hit (WorldEntity* entity, float damage)
{
  if (this->hitEntity != entity)
    this->destroy( entity );
  this->hitEntity = entity;
  //dynamic_cast<SpaceShip*>(entity)->damage(this->getPhysDamage(),this->getElecDamage());
 // this->destroy(this);
  this->deactivate();
}


void SwarmProjectile::setTarget(PNode* target)
{
    this->target = target;
}



/**
 *  this function gets called by tick to calculate the new flight direction
 *  @param curDirection direction vector
 *  @param estTargetDir target vector, pointing to where the target will be on hit
 *  @param angle = tick * turningSpeed
 *  @return (new) direction vector
*/
Vector SwarmProjectile::newDirection(Vector curDirection, Vector estTargetDir, float angle)
{
  if (unlikely(curDirection.len() == 0))
    return curDirection;
  //printf("recalculating direction\n");
  float tmp = angleRad ( curDirection, estTargetDir);
  if ( unlikely(tmp == 0) ) { return curDirection * maxVelocity / curDirection.len(); }
//   printf("turning angle: %f\ntemp: %f\n", angle, tmp);

  if( fabsf(angle) >  fabsf(tmp) )
    angle = tmp;
  else
    angle *= tmp/fabsf(tmp);

  Vector d = curDirection.cross(estTargetDir).cross(curDirection);
  d.normalize();
  if( unlikely( fabsf(angle) == 90)) { return d; } //avoid complication

  Vector newDir = curDirection + d *  curDirection.len() * tan (angle);
  newDir.normalize();
  newDir *= curDirection.len();
  return newDir;
}




/**
 *  signal tick, time dependent things will be handled here
 * @param time since last tick
*/
void SwarmProjectile::tick (float time)
{
  if(unlikely(this->target == NULL)) /** Check whether the target still exists*/
    this->deactivate();



/** old  guiding function*/

  float projectileVelocity = this->getVelocity().len();
  if (target != NULL){
    Vector estTargetDir = (this->target->getAbsCoor() - this->getAbsCoor()).getNormalized();
    this->velocity = this->newDirection(this->velocity, estTargetDir, this->turningSpeed * time );
  }
  else
    if (likely(projectileVelocity != 0 || projectileVelocity != this->maxVelocity) )
      this->velocity *= (this->maxVelocity / projectileVelocity); // set speed to max
/*
  printf("position: %f, %f, %f\n", this->getAbsCoor().x, this->getAbsCoor().y, this->getAbsCoor().z);
  printf("target position: %f, %f, %f\n", this->target->getAbsCoor().x, this->target->getAbsCoor().y, this->target->getAbsCoor().z);*/

  this->shiftCoor(this->velocity * time);

  if(this->tickLifeCycle(time))
    this->deactivate();

  this->trail->tick(time);

  this->angle += this->rotationSpeed * time;
  while (this->angle > 360)
    this->angle -= 360;

  this->lastDir = this->curDir;
  this->curDir = this->velocity;

  if( this->target != NULL && (this->getAbsCoor() - this->target->getAbsCoor()).len() < 3)   // HACK  Temp fake workaround for collision :)
  {
    dynamic_cast<WorldEntity*>(target)->destroy(this); //hit(this->getDamage(), this);
    this->deactivate();
    PRINTF(0)("Target was hit by Swarm Missile!\n");
  }
  else if( this->target == NULL)
    this->deactivate();
}

/**
 *  the function gets called, when the projectile is destroyed
 */
void SwarmProjectile::destroy (WorldEntity* killer)
{

//   printf("THIS SHOULD WORK!\n");

  Projectile::destroy( killer );
  PRINTF(5)("DESTROY SwarmProjectile\n");
  this->lifeCycle = .95; //!< @todo calculate this usefully.
  this->emitter->setSystem(SwarmProjectile::explosionParticles);

  this->emitter->setEmissionRate(1000.0);
  this->emitter->setEmissionVelocity(50.0);
  this->deactivate();

}


void SwarmProjectile::draw () const
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  Vector tmpDir = this->curDir; // *.7 + this->lastDir * .3;
  tmpDir.slerpTo(this->lastDir, .4);

  float matrix[4][4];
  glTranslatef (this->getAbsCoor ().x, this->getAbsCoor ().y, this->getAbsCoor ().z);
  Vector tmpRot = this->getAbsCoor().cross(tmpDir);
  glRotatef (angleRad ( this->getAbsCoor(), tmpDir), tmpRot.x, tmpRot.y, tmpRot.z );
  glRotatef(this->angle, 1.0f, 0.0f, 0.0f); //spinning missile
  this->getAbsDir().matrix (matrix);
  glMultMatrixf((float*)matrix);
  //glScalef(2.0, 2.0, 2.0);  // no double rescale
  this->getModel()->draw();
  glTranslatef(-.9,0,0);
  this->trail->draw();
  glPopMatrix();
}
