/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Patrick Boenzli, patrick@orxonox.net
   co-programmer:
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "executor/executor.h"
#include "util/loading/factory.h"
#include "util/loading/load_param.h"
#include "util/loading/load_param_xml.h"

#include "weapons/weapon_slot.h"

#include "particles/particle_system.h"

#include "mount_point.h"
#include "debug.h"
#include "state.h"


ObjectListDefinition(MountPoint);


/**
 * construct
 */
MountPoint::MountPoint (const Vector& up, const Vector& forward, const Vector& center, const std::string& name)
{
  PRINTF(0)("Created mount point %s\n", name.c_str());
  this->registerObject(this, MountPoint::_objectList);

  this->_name = name;
  this->setRelCoor( center);
  this->setRelDir( Quaternion(forward, up));

  this->init();
}



/**
 * deconstructor
 */
MountPoint::~MountPoint ()
{}


/**
 * initializing function
 */
void MountPoint::init()
{
  this->registerObject(this, MountPoint::_objectList);
  this->toList(OM_GROUP_00);

  this->_mount = NULL;
}


/**
 * loads the Settings of a MD2Creature from an XML-element.
 * @param root the XML-element to load the MD2Creature's properties from
 */
void MountPoint::initMountPoint(const TiXmlElement* root)
{
  if( root == NULL)
  {
    PRINTF(1)("MountPoint - initialization failed, since I got no valid xml element\n");
    return;
  }
  // now get the first element
  const TiXmlElement* element = root->FirstChildElement("MountPoints");
  if( element == NULL)
  {
    PRINTF(1)("I am in section: %s, Object Information file is missing a proper 'MountPoints' section\n", root->Value());
//     element = root->FirstChildElement( );
//     PRINTF(0)("first child: %s\n", element->Value());
  }
  else
  {
    element = element->FirstChildElement();
    // parse the information for this mount point

    PRINTF(4)("Loading WorldEntities\n");
    while( element != NULL)
    {
      std::string name = element->Value();

      PRINTF(5)("checking %s against local %s\n", name.c_str(), this->_name.c_str());
      // check if we got the right mount point
      if( this->_name.find(name, 0) != std::string::npos)
      {
        PRINTF(5)("found mount point %s\n", this->_name.c_str());
        // load it
        this->loadParam(element);
      }

      element = element->NextSiblingElement();
    }
  }
}



/**
 * load the parameters from the xml section
 * @param root the root element of this xml branche
 */
void MountPoint::loadParam(const TiXmlElement* root)
{
  // first check for the description
  LoadParam(root, "Description", this, MountPoint, setDescription)
  .describe("Sets this mount point a description");

  // now check for the orx class to create
  LoadParam(root, "OrxClass", this, MountPoint, setOrxClass)
  .describe("Sets the class this mount points should host");

  LoadParamXML(root, "Details", this, MountPoint, loadDetails);
}


/**
 * load the parameters from the world entity
 * @param root the root element of this xml branche
 */
void MountPoint::loadDetails(const TiXmlElement* root)
{
  if( this->_mount != NULL)
  {
    PRINTF(0)("Got detail informations\n");
    this->_mount->loadParams( root);

    if( this->_mount->isA(ParticleSystem::staticClassID()))
    {
      PRINTF(0)("got particle system. attaching it to mp\n");
      dynamic_cast<ParticleSystem*>(this->_mount)->attachEmmittersTo(this, Vector(0,0,0));
    }
  }
}


/**
 * setst the description (optional) via xml tag
 * @param description string containing the description
 */
void MountPoint::setDescription(const std::string& description)
{
  this->_description = description;
}



/**
 * sets the class of this mount point
 * @param orxClass class
 */
void MountPoint::setOrxClass(const std::string& orxClass)
{
  // create the object for this mount point
  BaseObject* obj = Factory::fabricate(orxClass);
  // check if the object is created correctly
  if( obj != NULL)
  {
    if( obj->isA( WorldEntity::staticClassID()))
    {
      PRINTF(0)("Mount Point created a %s\n", obj->getCName());
      // cast down the object to WE
      this->_mount = dynamic_cast<WorldEntity*>(obj);

      // now set the position, direction and reparent it to this node
      this->_mount->setAbsCoor( this->getAbsCoor());
      this->_mount->setAbsDir( this->getAbsDir());
      this->_mount->setParent( this);

      this->_mount->toList((OM_LIST)(this->getOMListNumber()));
    }
    else if( obj->isA( WeaponSlot::staticClassID()))
    {
      PRINTF(0)("=========+>we got a weapon slot\n");
    }
  }
  else
    PRINTF(1)("Couldn't create %s for this mount point (%s)\n", orxClass.c_str(), this->_name.c_str());
}



/**
 * tick
 * @param time  time passed since the last tick
 */
void MountPoint::tick (float time)
{

}


/**
 * draw this entity
 */
void MountPoint::draw() const
{
}



/**
 *  function called to draw the mount point itself for debug purposes only
 */
void MountPoint::debugDraw() const
{
  // invoke the underlying pnode debug draw
  this->debugDraw();
}


/**
 * adds an entity to this mount point
 * @param entity entity to be added
 */
void MountPoint::mount(WorldEntity* entity)
{
  this->_mount = entity;
}


/**
 * removes an entity from this mount point
 */
void MountPoint::unmount()
{
  this->_mount = NULL;
}

