/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: David Hasenfratz
*/

#include "billboard.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "graphics_engine.h"
#include "material.h"
#include "glincl.h"
#include "state.h"


#include "debug.h"


ObjectListDefinition(Billboard);
CREATE_FACTORY(Billboard);

/**
 * standart constructor
 */
Billboard::Billboard (const TiXmlElement* root)
{
  this->init();

  if( root)
    this->loadParams(root);
}


/**
 * destroys a Billboard
 */
Billboard::~Billboard ()
{
  if (this->material)
    delete this->material;
}


/**
 * initializes the Billboard
 */
void Billboard::init()
{
  this->registerObject(this, Billboard::_objectList);
  this->setName("Billboard");

  this->toList(OM_COMMON);

  this->material = new Material();
  this->setAbsCoor(0, 0, 0);
  //this->setVisibiliy(true);
  this->setSize(5, 5);

  this->texColor = NULL;
}


/**
 *  load params
 * @param root TiXmlElement object
 */
void Billboard::loadParams(const TiXmlElement* root)
{
  /*LoadParam(root, "texture", this->material, Material, setDiffuseMap)
      .describe("the texture-file to load onto the Billboard");

  LoadParam(root, "size", this, Billboard, setSize)
  .describe("the size of the Billboard in Pixels");*/
}


/**
 * sets the size of the Billboard.
 * @param size the size in pixels
 */
void Billboard::setSize(float sizeX, float sizeY)
{
  this->sizeX = sizeX;
  this->sizeY = sizeY;
}


/**
 * sets the material to load
 * @param textureFile The texture-file to load
 */
void Billboard::setTexture(const std::string& textureFile)
{
  this->material->setDiffuseMap(textureFile);
}


/**
 * ticks the Billboard
 * @param dt the time to ticks
 */
void Billboard::tick(float dt)
{
}


/**
 * draws the billboard
 */
void Billboard::draw() const
{
  if( !this->isVisible())
    return;

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);

  glPushMatrix();

  //glTranslatef(this->getAbsCoor().x, this->getAbsCoor().y, this->getAbsCoor().z);
  //glTranslatef(0,0,0);
  this->material->select();

  const PNode* camera = State::getCameraNode();  //!< @todo MUST be different
  Vector cameraPos = camera->getAbsCoor();
  Vector cameraTargetPos = State::getCameraTargetNode()->getAbsCoor();
  Vector view = cameraTargetPos - cameraPos;
  Vector up = Vector(0, 1, 0);
  up = camera->getAbsDir().apply(up);
  Vector h = up.cross(view);
  Vector v = h.cross(view);
  h.normalize();
  v.normalize();

  v *= sizeX;
  h *= sizeY;

//v += this->getAbsCoor();
    //PRINTF(0)("sizeX: %f sizeY: %f\n", sizeX, sizeY);

  // changes the color of the texture (if any is set)
  if(this->texColor != NULL)
    glColor4ub((GLubyte)(this->texColor->r()*255.0f), (GLubyte)(this->texColor->g()*255.0f), (GLubyte)(this->texColor->b()*255.0f), (GLubyte)(this->texColor->a()*255));

  glBegin(GL_QUADS);
  glTexCoord2f(0.0f, 0.0f);
  glVertex3f(this->getAbsCoor().x - h.x - v.x,
             this->getAbsCoor().y - h.y - v.y,
             this->getAbsCoor().z - h.z - v.z);
  glTexCoord2f(1.0f, 0.0f);
  glVertex3f( this->getAbsCoor().x + h.x - v.x,
              this->getAbsCoor().y + h.y - v.y,
              this->getAbsCoor().z + h.z - v.z);
  glTexCoord2f(1.0f, 1.0f);
  glVertex3f(this->getAbsCoor().x + h.x + v.x,
             this->getAbsCoor().y + h.y + v.y,
             this->getAbsCoor().z + h.z + v.z);
  glTexCoord2f(0.0f, 1.0f);
  glVertex3f(this->getAbsCoor().x - h.x + v.x,
             this->getAbsCoor().y - h.y + v.y,
             this->getAbsCoor().z - h.z + v.z);
  glEnd();


  glPopMatrix();

  glPopAttrib();
}


/**
 * changes the texture color
 * @param col color for the texture
 */
void Billboard::colorTexture(const Color col)
{
  this->texColor = new Color(col.r(), col.g(), col.b(), col.a());
}
