/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/


#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD

#include <fstream>
#include <iostream>

#include "game_world_data.h"

#include "state.h"
#include "substring.h"

#include "util/loading/game_loader.h"
#include "util/loading/resource_manager.h"

#include "world_entity.h"
#include "player.h"
#include "camera.h"
#include "cameraman.h"
#include "terrain.h"
#include "skybox.h"
#include "md2/md2Model.h"
#include "world_entities/projectiles/projectile.h"
#include "npcs/npc_test1.h"
#include "playable.h"

#include "light.h"

#include "util/loading/factory.h"
#include "loading/fast_factory.h"
#include "util/loading/load_param_xml.h"

#include "graphics_engine.h"
#include "graphics_effect.h"
#include "weather_effects/atmospheric_engine.h"
#include "event_handler.h"
#include "sound_engine.h"
#include "cd_engine.h"
#include "network_manager.h"
#include "physics_engine.h"

#include "glmenu_imagescreen.h"

#include "game_rules.h"

#include "ogg_player.h"
#include "shader.h"


ObjectListDefinition(GameWorldData);
/**
 * constructor of the GameWorldData
 */
GameWorldData::GameWorldData()
{
  this->registerObject(this, GameWorldData::_objectList);

  this->glmis = NULL;


  this->localPlayer = NULL;
  this->sky = NULL;
  this->terrain = NULL;

  this->music = NULL;
  this->objectManager = NULL;
  this->gameRule = NULL;
}


/**
 * destructor for the GameWorldData
 */
GameWorldData::~GameWorldData()
{}



/**
 *  initialize the GameWorldData
 */
ErrorMessage GameWorldData::init()
{
  this->objectManager = new ObjectManager();
  State::setObjectManager(this->objectManager);

  PNode::getNullParent();
  Camera* localCamera = new Camera();
    localCamera->setName ("GameWorld-Camera");
  State::setCamera(localCamera, localCamera->getTarget());
  //CameraMan* camMan = new CameraMan();
  //State::setCameraman(camMan);
  LightManager::getInstance();

//  GraphicsEngine::getInstance()->displayFPS(true);

  return ErrorMessage();
}


/**
 *  loads the data from the xml file
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadData(const TiXmlElement* root)
{
  assert (root != NULL);
  // load the parameters
  // name
  std::string string = LoadParamBase::grabParameter( root, "name");
  if( string.empty() )
  {
    PRINTF(2)("GameWorld is missing a proper 'name'\n");
    this->setName("Unknown");
  }
  else
    this->setName(string.c_str());

  this->loadGUI(root);
  this->loadWorldEntities(root);
  this->loadCameras(root);
  this->loadScene(root);

  return ErrorMessage();
}


/**
 *  unloads the data from the xml file
 */
ErrorMessage GameWorldData::unloadData()
{
  this->unloadGUI();
  this->unloadWorldEntities();
  this->unloadScene();
  this->unloadCameras();

  return ErrorMessage();
}

/**
 * @brief creates the cameras
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadCameras(const TiXmlElement* root)
{
  const TiXmlElement* element = root->FirstChildElement("CameraMan");
  if( element != NULL)
  {
    printf("found CameraMan tag  \n");
    CameraMan* camMan = new CameraMan(element);
    State::setCameraman(camMan);
  }
  return ErrorMessage();
}


/**
 * @brief unloads the camera
 */
ErrorMessage GameWorldData::unloadCameras()
{
  CameraMan* camMan = State::getCameraman() ;
  if(camMan != NULL){
    delete  camMan;
    State::setCameraman(NULL);
   }
  return ErrorMessage();
}


/**
 * @brief loads the GUI data
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadGUI(const TiXmlElement* root)
{
  const TiXmlElement* element = root->FirstChildElement("LoadScreen");
  if( element == NULL)
  {
    PRINTF(2)("no LoadScreen specified, loading default\n");

    glmis->setBackgroundImage("textures/load_screens/default.jpg");
    this->glmis->setMaximum(8);
    //     this->glmis->draw();
  }
  else
  {
    this->glmis->loadParams(element);
    //     this->glmis->draw();
  }
  this->glmis->draw();

  return ErrorMessage();
}


/**
 * @brief unloads the GUI data
 */
ErrorMessage GameWorldData::unloadGUI()
{
  delete this->glmis;

  return ErrorMessage();
}


/**
 * @brief loads the world entities from the xml file
 * @param root reference to the xml root parameter
 */
ErrorMessage GameWorldData::loadWorldEntities(const TiXmlElement* root)
{
  const TiXmlElement* element = root->FirstChildElement("WorldEntities");
  bool mouseCaptured = EventHandler::getInstance()->grabbedEvents();
  EventHandler::getInstance()->grabEvents(false);

  if( element == NULL)
  {
    PRINTF(1)("GameWorld is missing 'WorldEntities'\n");
  }
  else
  {
    element = element->FirstChildElement();
    // load Players/Objects/Whatever
    PRINTF(4)("Loading WorldEntities\n");
    while( element != NULL)
    {
      BaseObject* created = Factory::fabricate(element);
      if( created != NULL )
        PRINTF(4)("Created a %s: %s\n", created->getClassCName(), created->getCName());

      //todo do this more elegant
      if( element->Value() == "SkyBox" && created->isA(SkyBox::staticClassID()))
      {
        this->sky = dynamic_cast<WorldEntity*>(created);
        State::setSkyBox(dynamic_cast<SkyBox*>(this->sky));
      }
      if( element->Value() == "Terrain" && created->isA(Terrain::staticClassID()))
      {
        this->terrain = dynamic_cast<Terrain*>(created);
        CDEngine::getInstance()->setTerrain(terrain);
      }
      element = element->NextSiblingElement();
      this->glmis->step(); //! @todo temporary
    }
    PRINTF(4)("Done loading WorldEntities\n");
  }

  // Create a Player
  this->localPlayer = new Player();
  State::setPlayer(this->localPlayer);

  if (!Playable::objectList().empty())
  {
    /// TODO Make this also loadable
    this->localPlayer->setPlayable(Playable::objectList().front());
  }

  // Fill the EntityLists. Tick then Draw:
  this->tickLists.push_back(OM_DEAD_TICK);
  this->tickLists.push_back(OM_ENVIRON);
  this->tickLists.push_back(OM_COMMON);
  this->tickLists.push_back(OM_GROUP_00);
  this->tickLists.push_back(OM_GROUP_00_PROJ);
  this->tickLists.push_back(OM_GROUP_01);
  this->tickLists.push_back(OM_GROUP_01_PROJ);
  this->tickLists.push_back(OM_GROUP_02);

  this->drawLists.push_back(OM_ENVIRON_NOTICK);
  this->drawLists.push_back(OM_ENVIRON);
  this->drawLists.push_back(OM_GROUP_00);
  this->drawLists.push_back(OM_GROUP_00_PROJ);
  this->drawLists.push_back(OM_GROUP_01);
  this->drawLists.push_back(OM_GROUP_01_PROJ);
  this->drawLists.push_back(OM_GROUP_02);
  this->drawLists.push_back(OM_COMMON);

  /* init the pnode tree */
  PNode::getNullParent()->init();

  EventHandler::getInstance()->grabEvents(mouseCaptured);

  return ErrorMessage();
}


/**
 *  unloads the world entities
 */
ErrorMessage GameWorldData::unloadWorldEntities()
{
  FastFactory::flushAll(true);
  GraphicsEngine::getInstance()->displayFPS(false);
  // erease everything that is left.
  //secondary cleanup of PNodes;
  while (!PNode::objectList().empty())
    delete PNode::objectList().front();

  /* remove the player object */
  if( this->localPlayer)
    delete this->localPlayer;
  State::setPlayer(NULL);
  this->localPlayer = NULL;
  State::setCamera(NULL, NULL);
  this->sky = NULL;
  this->terrain = NULL;

  while (!GraphicsEffect::objectList().empty())
    delete GraphicsEffect::objectList().front();


  while (!Element2D::objectList().empty())
    delete Element2D::objectList().front();

  // At this Point all the WorldEntites should be unloaded.
  this->tickLists.clear();
  this->drawLists.clear();

  // unload the resources loaded in this Level !!
  /// FIXME TODO HACK!!
//  ResourceManager::getInstance()->unloadAllByPriority(RP_LEVEL);

  if (State::getObjectManager() == this->objectManager)
  {
    State::setObjectManager(NULL);
    delete this->objectManager;
  }
  this->objectManager = NULL;

  if(State::getSkyBox())
    State::setSkyBox(NULL);

  this->glmis = NULL;

  return ErrorMessage();
}


/**
 * @brief loads the scene data
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadScene(const TiXmlElement* root)
{
  LoadParamXML(root, "LightManager", LightManager::getInstance(), LightManager, loadParams);
  LoadParamXML(root, "GraphicsEngine", GraphicsEngine::getInstance(), GraphicsEngine, loadParams);
  LoadParamXML(root, "AtmosphericEngine", AtmosphericEngine::getInstance(), AtmosphericEngine, loadParams);

  LoadParam(root, "Music", this, GameWorldData, setSoundTrack);

  LoadParamXML(root, "GameRule", this, GameWorldData, loadGameRule);

  LoadParam(root, "clip-region", State::getCameraman(), CameraMan, setClipRegion);


  //LoadParamXML(root, "ParticleEngine", ParticleEngine::getInstance(), ParticleEngine, loadParams);
  //LoadParamXML(root, "PhysicsEngine", PhysicsEngine::getInstance(), PhysicsEngine, loadParams);


  OrxSound::SoundEngine::getInstance()->setListener(State::getCamera());

  return ErrorMessage();
}



/**
 *  unloads the scene data
 */
ErrorMessage GameWorldData::unloadScene()
{
  /* delete some garphics and scene eingines */
  delete LightManager::getInstance();
  delete AtmosphericEngine::getInstance();

  if (this->music != NULL)
    this->setSoundTrack("");

  /* unload the shaders */
  Shader::suspendShader();

  State::setGameRules(NULL);

  return ErrorMessage();
}


void GameWorldData::setSoundTrack(const std::string& name)
{
  if (this->music != NULL)
    delete this->music;
  this->music = NULL;

  if (!name.empty())
  {
    PRINTF(3)("Setting Sound Track to %s\n", name.c_str());
    std::string oggFile = Resources::ResourceManager::getInstance()->prependAbsoluteMainPath(name);
    this->music = new OrxSound::OggPlayer(oggFile);
    if (this->localPlayer != NULL)
      this->localPlayer->hud().notifyUser(std::string("Playing SoundTrack: ") + this->music->artist() + " - " + this->music->title());

    //(OggPlayer*)ResourceManager::getInstance()->load(name, OGG, RP_LEVEL);
    //assert(this->music->isA(CL_SOUND_OGG_PLAYER));
  }
}


void GameWorldData::loadGameRule(const TiXmlElement* root)
{
  const TiXmlElement* element = root->FirstChildElement();
  while( element != NULL)
  {
    PRINTF(2)("============ GameRules ==\n");
    PRINTF(2)("creating %s\n", element->Value());
    BaseObject* created = Factory::fabricate(element);
    if (created != NULL && created->isA(GameRules::staticClassID()))
    {
      this->gameRule = dynamic_cast<GameRules*>(created);
      State::setGameRules(this->gameRule);
      // if there is a valid game rule loaded, return because it is not thought to load multiple game rules
      return;
    }
    else
    {
      PRINTF(1)("Could not create a %s\n", element->Value());
      delete created;
    }
    element = element->NextSiblingElement();
  }
}



