/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "planet.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "static_model.h"

#include "material.h"
#include "texture.h"

#include "network_game_manager.h"
#include "converter.h"
#include "vertex_array_model.h"
#include "primitive_model.h"

#include "debug.h"


ObjectListDefinition(Planet);
CREATE_FACTORY(Planet);


/**
 *  initializes a skybox from a XmlElement
*/
Planet::Planet(const TiXmlElement* root)
{
  this->registerObject(this, Planet::_objectList);
  this->toList(OM_GROUP_01);

  this->rotSpeedPlanet = 0.0;
  this->rotSpeedCloud = 0.0;

  //this->materialPlanet->setIllum(20);
  //this->materialPlanet->setAmbient(0.1, 0.1, 0.1);

  if( root != NULL)
    this->loadParams(root);

  PrimitiveModel* model = new PrimitiveModel(PRIM_SPHERE, this->size, 50);
  this->setModel(model);

  this->cloudModel = new PrimitiveModel(PRIM_SPHERE, this->size + 10, 50);
}


/**
 *  default destructor
*/
Planet::~Planet()
{
  PRINTF(5)("Deleting Planet\n");
}


void Planet::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "texture", this, Planet, setTexture)
      .describe("Sets the materialPlanet on the Planet. The string must be the path relative to the data-dir, and without a trailing .jpg");

  LoadParam(root, "cloud-texture", this, Planet, setCloudTexture)
      .describe("Sets the cloud texture of the planet");

  LoadParam(root, "cloud-rotation", this, Planet, setCloudRotation)
      .describe("Sets the cloud rotation speed");

  LoadParam(root, "size", this, Planet, setSize)
      .describe("Sets the Size of the Planet (normally this should be 90% of the maximal viewing Distance).");
}


/**
 *  Defines which textures should be loaded onto the Planet.
 * @param textureName the top texture.
*/
void Planet::setTexture(const std::string& textureName)
{
  this->materialPlanet.setDiffuseMap(textureName);
}


/**
 *  Defines which textures should be loaded onto the Planet.
 * @param textureName the top texture.
*/
void Planet::setCloudTexture(const std::string& textureName)
{
  this->materialCloud.setDiffuseMap(textureName);
}


void Planet::setCloudRotation(float rotationSpeed)
{
  this->rotSpeedCloud = rotationSpeed;
}

/**
 * @param size The new size of the Planet

 * do not forget to rebuild the Planet after this.
*/
void Planet::setSize(float size)
{
  this->size = size;
}


void Planet::tick( float dt)
{
  if( dt != 0.)
    this->shiftDir( Quaternion( this->rotSpeedPlanet * dt, Vector(1,0,0)));
}


/**
 * draws the planet
 */
void Planet::draw() const
{
  // draw the clouds
  this->materialPlanet.select();
  WorldEntity::draw();

  this->materialCloud.select();
  WorldEntity::draw(this->cloudModel);

}



