/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004-2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Nicolas Schlumberger, Marc Schaerrer
   co-programmer: Benjamin Grauer

*/


#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WEAPON

#include "spike_ball.h"

#include "state.h"
#include "model.h"

#include "particles/dot_emitter.h"
#include "particles/sprite_particles.h"

#include <cassert>
#include "debug.h"

#include "space_ships/space_ship.h"



ObjectListDefinition(SpikeBall);
CREATE_FAST_FACTORY_STATIC(SpikeBall);

/**
 *  standard constructor
*/
SpikeBall::SpikeBall () : ProjectileWeapon()
{
  this->registerObject(this, SpikeBall::_objectList);

  this->loadModel("models/projectiles/spike_ball.obj", .25);

  this->setMinEnergy(1);
  this->setHealthMax(1);
  this->lifeSpan = 1.0;

  this->emitter = new DotEmitter(100, 5, M_2_PI);
  this->emitter->setParent(this);
  this->emitter->setSpread(M_PI, M_PI);
  this->emitter->setEmissionRate(300.0);
  this->emitter->setEmissionVelocity(50.0);

  this->setRotationSpeed(130);

  this->halo = new Billboard();
  this->halo->setSize(2, 2);
  this->halo->setTexture("hbolt_halo.png");

  this->setFragments(26);

  this->size = 4;

  this->launcher = new Vector [this->getFragments()];
}


/**
 *  standard deconstructor
*/
SpikeBall::~SpikeBall ()
{
  // delete this->emitter;

  /* this is normaly done by World.cc by deleting the ParticleEngine */
  if (SpikeBall::explosionParticles != NULL && SpikeBall::objectList().size() <= 1)
  {
    //if (ClassList::exists(SpikeBall::explosionParticles, CL_PARTICLE_SYSTEM))
    //  delete SpikeBall::explosionParticles;
    PRINTF(1)("Deleting SpikeBall Particles\n");
//    if (SpikeBall::objectList().exists(SpikeBall::explosionParticles))
//      delete SpikeBall::explosionParticles;
    SpikeBall::explosionParticles = NULL;
  }

}

SpriteParticles* SpikeBall::explosionParticles = NULL;

void SpikeBall::activate()
{
  if (unlikely(SpikeBall::explosionParticles == NULL))
  {
    SpikeBall::explosionParticles = new SpriteParticles(1000);
    SpikeBall::explosionParticles->setName("SpikeBallExplosionParticles");
    SpikeBall::explosionParticles->setLifeSpan(.5, .3);
    SpikeBall::explosionParticles->setRadius(0.0, 10.0);
    SpikeBall::explosionParticles->setRadius(.5, 6.0);
    SpikeBall::explosionParticles->setRadius(1.0, 3.0);
    SpikeBall::explosionParticles->setColor(0.0, 1,1,0,.9);
    SpikeBall::explosionParticles->setColor(0.5, .8,.8,0,.5);
    SpikeBall::explosionParticles->setColor(1.0, .8,.8,.7,.0);
  }
  this->setDamage(5);
  this->setHealth(10);
  this->setRotationAxis(VECTOR_RAND(1));
  this->setAngle();

  this->launcher[0] = Vector(1.0, 0.0, 0.0);
  this->launcher[1] = Vector(0.0, 1.0, 0.0);
  this->launcher[2] = Vector(0.0, 0.0, 1.0);

  this->launcher[3] = Vector(1.0, 1.0, 0.0);
  this->launcher[4] = Vector(0.0, 1.0, 1.0);
  this->launcher[5] = Vector(1.0, 0.0, 1.0);
  this->launcher[6] = Vector(1.0, -1.0, 0.0);
  this->launcher[7] = Vector(0.0, 1.0, -1.0);
  this->launcher[8] = Vector(-1.0, 0.0, 1.0);

  this->launcher[9] = Vector(-1.0, 1.0, 1.0);
  this->launcher[10] = Vector(1.0, 1.0, 1.0);
  this->launcher[11] = Vector(1.0, -1.0, 1.0);
  this->launcher[12] = Vector(-1.0, -1.0, 1.0);

  int tmp = this->getFragments() / 2;
  for (int i = 0; i < tmp; i++)
  {
    this->launcher[i].normalize();
    this->launcher[tmp + i] =  this->launcher[i] * (-1);
  }

  this->freeMode = false;
}


void SpikeBall::deactivate()
{
  assert (SpikeBall::explosionParticles != NULL);
  //SpikeBall::explosionParticles->removeEmitter(this->emitter);
  this->emitter->setSystem(NULL);
  this->lifeCycle = 0.0;

  this->toList(OM_NULL);
  this->removeNode();
  SpikeBall::fastFactory->kill(this);
}

/*
void SpikeBall::collidesWith(WorldEntity* entity, const Vector& location)
{
  PRINTF(0)("Collision with SpikeBall\n");
  if (this->hitEntity != entity && entity->isA(CL_NPC))
    this->destroy( entity );
  this->hitEntity = entity;
  dynamic_cast<SpaceShip*>(entity)->damage(this->getDamage(),0);
}*/


void SpikeBall::blow()
{
  if ( this->freeMode )
    updateFireDir();

  Spike* pj = NULL;
  for ( int i = 0; i < this->getFragments(); i++)
  {
    pj  = new Spike();
    assert( pj );
    pj->setParent(PNode::getNullParent());

    pj->setVelocity(this->launcher[i].getNormalized() * 250.0);

    pj->setParent(PNode::getNullParent());
    pj->setAbsCoor(this->getAbsCoor() + this->launcher[i] * this->size);
//     Quaternion q;
    pj->setAbsDir(Quaternion(this->launcher[i], 0));

    pj->toList(this->getOMListNumber());

    pj->activate();
  }
}


void SpikeBall::updateFireDir(){

  float** m = new float* [3];
  for( int i = 0; i < 3 ; i++)
    m[i] = new float;

  float nx, ny, nz, ca, sa;

  nx = this->getRotationAxis().x;
  ny = this->getRotationAxis().y;
  nz = this->getRotationAxis().z;

  ca = cos (this->getAngle());
  sa = sin (this->getAngle());

  m[0][0] = nx * nx * (1 - ca) + ca;
  m[0][1] = nx * ny * (1 - ca) + nz * sa;
  m[0][2] = nx * nz * (1 - ca) - ny * sa;
  m[1][0] = nx * nz * (1 - ca) - nz * sa;
  m[1][1] = ny * ny * (1 - ca) + ca;
  m[1][2] = ny * nz * (1 - ca) + nx * sa;
  m[2][0] = nx * nz * (1 - ca) + ny * sa;
  m[2][1] = ny * nz * (1 - ca) - nx * sa;
  m[2][2] = nz * nz * (1 - ca) + ca;

  float x, y, z;
  for (int i = 0; i < this->getFragments(); i++){
    x = m[0][0] * this->launcher[i].x + m[0][1] * this->launcher[i].y + m[0][2] * this->launcher[i].z;
    y = m[1][0] * this->launcher[i].x + m[1][1] * this->launcher[i].y + m[1][2] * this->launcher[i].z;
    z = m[2][0] * this->launcher[i].x + m[2][1] * this->launcher[i].y + m[2][2] * this->launcher[i].z;

    this->launcher[i] = Vector (x, y, z);
  }

  for( int i = 0; i < 3 ; i++)
    delete m[i];
  delete m;
}


/**
 *  signal tick, time dependent things will be handled here
 * @param dt time since last tick
*/
void SpikeBall::tick (float dt)
{
  Vector v = this->velocity * dt;
  this->shiftCoor(v);

  if (this->tickLifeCycle(dt)){
    this->blow();
    this->deactivate();
  }

  this->updateAngle( dt );
}

/**
 *  the function gets called, when the projectile is destroyed
*/
void SpikeBall::destroy (WorldEntity* killer)
{
  ProjectileWeapon::destroy( killer );
  PRINTF(5)("DESTROY SpikeBall\n");
  this->lifeCycle = .95; //!< @todo calculate this usefully.

  this->emitter->setSystem(SpikeBall::explosionParticles);
}


void SpikeBall::draw () const
{
  glPushAttrib(GL_ENABLE_BIT);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  float matrix[4][4];
  glTranslatef (this->getAbsCoor ().x, this->getAbsCoor ().y, this->getAbsCoor ().z);
  this->halo->draw();

  glRotatef(angle, this->getRotationAxis().x, this->getRotationAxis().y, this->getRotationAxis().z);
  this->getAbsDir().matrix (matrix);
  glMultMatrixf((float*)matrix);
  this->getModel()->draw();


  glPopMatrix();
  glPopAttrib();
}

