/*!
 * @file projectile_weapon.h
 * a projectile_weapon, that is been shooted by a weapon
 *
 * You can use this class to make some ProjectileWeapons/Bullets/Lasers/Rockets/etc.
 *
 */

#ifndef _PROJECTILE_WEAPON_H
#define _PROJECTILE_WEAPON_H

#include "world_entity.h"
#include "loading/fast_factory.h"
#include "space_ships/space_ship.h"

#include "loading/fast_factory.h"
#include "world_entities/projectiles/projectile.h"
#include "weapons/weapon.h"


#include "sound_source.h"
#include "sound_buffer.h"

class FastFactory;


class ProjectileWeapon : public Projectile
{
  ObjectListDeclaration(ProjectileWeapon);
  public:
    ProjectileWeapon ();
    virtual ~ProjectileWeapon ();

    /** @brief Constructor with variable passing*/
    ProjectileWeapon (float pDamage, float eDamage, PNode* target);
    /** @brief for void construction; setting values later - needed for FastFactory*/
    virtual void initialize(float pDamage, float eDamage, PNode* target);

    void setFlightDirection(const Quaternion& flightDirection);
    void setVelocity(const Vector &velocity);
    void setLifeSpan(float lifeSpan);

    void loadExplosionSound(const std::string& explosionSound);
    void loadEngineSound(const std::string& engineSound);
    void setMinEnergy(float energyMin);
    /** @returns the minimal charched energy */
    inline float getMinEnergy() { return this->energyMin; };
    /** @returns if the ProjectileWeapon can be charged */
    inline bool isChageable() { return this->bChargeable; };

    void setTarget(PNode* target);

    /** @brief This is called, when the ProjectileWeapon is Emitted */
    virtual void activate() = 0;
    /** @brief This is called, when the ProjectileWeapon is being destroyed, or deleted */
    virtual void deactivate() = 0;

    virtual void destroy (WorldEntity* killer);

    virtual void collidesWith (WorldEntity* target, const Vector& location);  //!< collision handler; used against SpaceShip as most target will be


    virtual void tick (float dt);
    /** @brief convenience function
     * @param dt the Time passed
     * @returns true if the ProjectileWeapon is past its lifeTime, false if it shall still live */
    inline bool tickLifeCycle(float dt ) { this->lifeCycle += dt/this->lifeSpan;  return(unlikely(this->lifeCycle >= 1)); }

    inline float getPhysDamage() { return this->physDamage; };
    inline float getElecDamage() { return this->elecDamage; };

    inline void setPhysDamage( float dmg) {this->physDamage = dmg; };
    inline void setElecDamage( float dmg) {this->elecDamage = dmg; };

    inline void setRotationAxis ( Vector axis ) { this->rotationAxis = axis; };
    inline void setRotationSpeed ( float speed ) { this->rotationSpeed = speed; };
    inline const Vector & getRotationAxis () const { return this->rotationAxis; };
    inline float getRotationSpeed () { return this->rotationSpeed; };     //!< Added for completeness

    inline void setAngle () { this->angle = 0; };
    inline void setAngle ( float angle ) { this->angle = angle; };
    inline float getAngle () { return this->angle; };

    inline void updateAngle ( float dt ) { this->angle += this->rotationSpeed * dt; };

    inline void setFragments( int frag ) { this->fragments = frag; };
    inline int getFragments () { return this->fragments; };

//     virtual void blow ();

  protected:
    // energy
    float                   energyMin;                //!< The minimal Energy a ProjectileWeapon needs to be emitted.
    bool                    bChargeable;              //!< if the ProjectileWeapon is Charegeable

    float                   lifeCycle;                //!< The percentage of the Lifetime done [0-1]
    float                   lifeSpan;                 //!< The entire lifespan of the Shoot. in seconds

    float                   physDamage;               //!< damage to shield and armor
    float                   elecDamage;               //!< damage to elctronic
    float                   turningSpeed;             //!< degrees per tick

    Vector                  flightDirection;          //!< DOF direction in which the shoot flighs

    float                   angle;                    //!< Spinning Projectile, needed to ajust dispersal pattern
    float                   rotationSpeed;            //!< rotation speed
    Vector                  rotationAxis;           //!< rotation axis

    int                     fragments;                //!< Number of fragements created by the blow

    Vector                  velocity;                 //!< velocity of the projectile_weapon.

    PNode*                  target;                   //!< A target for guided Weapons.

    virtual void            blow();

//     FastFactory             ff;
//     static FastFactory*               fastFactory;

    OrxSound::SoundSource  soundSource;
  private:
    OrxSound::SoundBuffer  explosionBuffer;
    OrxSound::SoundBuffer  engineBuffer;
};

#endif /* _PROJECTILE_WEAPON_H */
