/*!
 * @file projectile.h
 * a projectile, that is been shooted by a weapon
 *
 * You can use this class to make some Projectiles/Bullets/Lasers/Rockets/etc.
 *
 */

#ifndef _PROJECTILE_H
#define _PROJECTILE_H

#include "world_entity.h"
#include "loading/fast_factory.h"
#include "space_ships/space_ship.h"

#include "sound_source.h"
#include "sound_buffer.h"

class Projectile : public WorldEntity
{
  ObjectListDeclaration(Projectile);
  public:
    Projectile ();
    virtual ~Projectile ();

    /** @brief Constructor with variable passing*/
    Projectile (float pDamage, float eDamage, PNode* target);
    /** @brief for void construction; setting values later - needed for FastFactory*/
    virtual void initialize(float pDamage, float eDamage, PNode* target);

    void setFlightDirection(const Quaternion& flightDirection);
    void setVelocity(const Vector &velocity);
    void setLifeSpan(float lifeSpan);

    void loadExplosionSound(const std::string& explosionSound);
    void loadEngineSound(const std::string& engineSound);
    void setMinEnergy(float energyMin);
    /** @returns the minimal charched energy */
    inline float getMinEnergy() { return this->energyMin; };
    /** @returns if the Projectile can be charged */
    inline bool isChageable() { return this->bChargeable; };

    void setTarget(PNode* target);

    /** @brief This is called, when the Projectile is Emitted */
    virtual void activate() = 0;
    /** @brief This is called, when the Projectile is being destroyed, or deleted */
    virtual void deactivate() = 0;

    virtual void destroy (WorldEntity* killer);

    virtual void collidesWith (WorldEntity* target, const Vector& location);  //!< collision handler; used against SpaceShip as most target will be


    virtual void tick (float dt);
    /** @brief convenience function
     * @param dt the Time passed
     * @returns true if the Projectile is past its lifeTime, false if it shall still live */
    inline bool tickLifeCycle(float dt ) { this->lifeCycle += dt/this->lifeSpan;  return(unlikely(this->lifeCycle >= 1)); }

    inline float getPhysDamage() { return this->physDamage; };
    inline float getElecDamage() { return this->elecDamage; };

    inline void setPhysDamage( float dmg) {this->physDamage = dmg; };
    inline void setElecDamage( float dmg) {this->elecDamage = dmg; };

  protected:
    // energy
    int		       origList;			//!< FIXME currently a fix around the collision seg fault
    float                   energyMin;                //!< The minimal Energy a Projectile needs to be emitted.
    bool                    bChargeable;              //!< if the Projectile is Charegeable

    float                   lifeCycle;                //!< The percentage of the Lifetime done [0-1]
    float                   lifeSpan;                 //!< The entire lifespan of the Shoot. in seconds

    float                   physDamage;               //!< damage to shield and armor
    float                   elecDamage;               //!< damage to elctronic
    float                   turningSpeed;             //!< degrees per tick

    Vector                  flightDirection;          //!< DOF direction in which the shoot flighs

    Vector                  velocity;                 //!< velocity of the projectile.

    PNode*                  target;                   //!< A target for guided Weapons.

    OrxSound::SoundSource  soundSource;
  private:
    OrxSound::SoundBuffer  explosionBuffer;
    OrxSound::SoundBuffer  engineBuffer;
};

#endif /* _PROJECTILE_H */
