
/*!
 * @file playable.h
 * Interface for a basic controllable WorldEntity
 */
#ifndef _PLAYABLE_H
#define _PLAYABLE_H

#include "world_entity.h"
#include "extendable.h"
#include "event.h"
#include <vector>
#include <list>

#include "world_entities/weapons/weapon_manager.h"

class Weapon;
class DotEmitter;
class Player;
class SpriteParticles;
class Explosion;

//! Basic controllable WorldEntity
/**
 *
 */
class Playable : public WorldEntity, public Extendable
{
  ObjectListDeclaration(Playable);
public:
  //! Defines the Playmode of an Entity.
  typedef enum {
    Vertical         =  1,       //!< Vertical (seen from left or right/move in x-z)
    Horizontal       =  2,       //!< Horizontal (seet from the top/move in x-y)
    FromBehind       =  4,       //!< Seen from behind (move in z-y)
    Full3D           =  8,       //!< Full featured 3D-mode. (move in all directions x-y-z)
    FirstPerson      = 16,

    PlaymodeCount    =  5,
  } Playmode;


public:
  virtual ~Playable();

  virtual void loadParams(const TiXmlElement* root);
  void varChangeHandler( std::list< int > & id );

  // Weapon and Pickups
  virtual bool pickup(PowerUp* powerUp);
  bool addWeapon(Weapon* weapon, int configID = -1, int slotID = -1);
  void removeWeapon(Weapon* weapon);
  void nextWeaponConfig();
  void previousWeaponConfig();
  inline WeaponManager& getWeaponManager() { return this->weaponMan; };
  void weaponConfigChanged();
  static void addSomeWeapons_CHEAT();


  // Player Settup
  bool hasPlayer(){return !(this->currentPlayer == NULL);}
  bool setPlayer(Player* player);
  Player* getCurrentPlayer() const { return this->currentPlayer; };
  /** @return a List of Events in PEV_* sytle */
  inline const std::vector<int>& getEventList() { return this->events; };


  // Camera and Playmode
  void attachCamera();
  void detachCamera();
  void setCameraMode(unsigned int cameraMode = 0);
  bool playmodeSupported(Playable::Playmode playmode) const { return this->supportedPlaymodes & playmode; };
  virtual bool setPlaymode(Playable::Playmode playmode);
  Playable::Playmode getPlaymode() const { return this->playmode; };
  virtual void setPlayDirection(const Quaternion& rot, float speed = 0.0f) = 0;
  void setPlayDirection(float angle, float dirX, float dirY, float dirZ, float speed = 0.0f);

  // Networking interface
  inline void setScore( int score ) { this->score = score; }
  inline int  getScore() { return this->score; }
  inline void setTeamId( int teamId) { this->teamId = teamId;}
  inline int getTeamId() const { return this->teamId; }
  virtual void setTeam(int teamID);


  void setEnterRadius(float radius) { this->enterRadius = radius; };
  /** @returns the EnterRadius (how far away a Player must be to enter this entity) */
  inline float getEnterRadius() const { return this->enterRadius; };

  // WorldEntity Extensions
  virtual void destroy(WorldEntity* killer);
  virtual void respawn();
  virtual void process(const Event &event);
  virtual void tick(float dt);


  inline bool beFire() const { return this->bFire; }
  inline void fire(bool bF) { this->bFire = bF;}

  // Transformations:
  static Playable::Playmode stringToPlaymode(const std::string& playmode);
  static const std::string& playmodeToString(Playable::Playmode playmode);
  static const std::string playmodeNames[];

protected:
  Playable();

  // Player Setup
  virtual void enter() = 0;
  virtual void leave() = 0;
  // Playmode
  void setSupportedPlaymodes(short playmodes) { this->supportedPlaymodes = playmodes; };
  virtual void enterPlaymode(Playable::Playmode playmode);

  // Events.
  void registerEvent(int eventType);
  void unregisterEvent(int eventType);

protected:
  WeaponManager         weaponMan;          //!< the weapon manager: managing a list of weapon to wepaon-slot mapping
  bool                  bFire;              //!< If the Ship is firing.
  short                 supportedPlaymodes; //!< What Playmodes are Supported in this Playable.
  Playable::Playmode    playmode;           //!< The current playmode.

private:
  std::vector<int>      events;             //!< A list of Events, that are captured for this playable

  Player*               currentPlayer;      //!< The Player currently connected to this Playable (the one that has controll) otherwise NULL

  int                   oldFlags;           //!< Used for synchronisation

  int                   score;              //!< players score
  int                   teamChangeHandler;  //!< handler id for team changes network sync
  int                   teamId;             //!< id of the current team

  bool                  bDead;

  float                 enterRadius;        //!< How far one can be away from the Playable to enter it.

  WorldEntity*          collider;
};

#endif /* _PLAYABLE_H */
