
#ifndef __Spline_h
#define __Spline_h

#include "vtkObject.h"

class Spline
{
public:

  // Description:
  // Set/Get the parametric range. If not set, the range is determined
  // implicitly by keeping track of the (min,max) parameter values for
  // t. If set, the AddPoint() method will clamp the t value to lie
  // within the specified range.
  void SetParametricRange(double tMin, double tMax);
  void SetParametricRange(double tRange[2])
    {this->SetParametricRange(tRange[0],tRange[1]);}
  void GetParametricRange(double tRange[2]) const;


  // Description:
  // Compute the coefficients for the spline.
  virtual void Compute () = 0;

  // Description:
  // Interpolate the value of the spline at parametric location of t.
  virtual double Evaluate (double t) = 0;

  // Description:
  // Return the number of points inserted thus far.
  int GetNumberOfPoints();

  // Description:
  // Add a pair of points to be fit with the spline.
  void AddPoint (double t, double x);

  // Description:
  // Remove a point from the data to be fit with the spline.
  void RemovePoint (double t);

  // Description:
  // Remove all points from the data.
  void RemoveAllPoints ();



protected:
  Spline();
  ~Spline();

  unsigned long ComputeTime;
  // Description:
  // Set/Get ClampValue. If On, results of the interpolation will be
  // clamped to the min/max of the input data.
  int ClampValue;
  double *Intervals;
  double *Coefficients;
  int LeftConstraint;
  // Description:
  // The values of the derivative on the left and right sides. The value
  // is used only if the left(right) constraint is type 1-3.
  double LeftValue;
  // Description:
  // Set the type of constraint of the left(right) end points. Four
  // constraints are available:
  //
  // 0: the first derivative at left(right) most point is determined
  // from the line defined from the first(last) two points.
  //
  // 1: the first derivative at left(right) most point is set to
  // Left(Right)Value.
  //
  // 2: the second derivative at left(right) most point is set to
  // Left(Right)Value.
  //
  // 3: the second derivative at left(right)most points is Left(Right)Value
  // times second derivative at first interior point.
  int RightConstraint;
  double RightValue;
  vtkPiecewiseFunction *PiecewiseFunction;
  // Description:
  // Control whether the spline is open or closed. A closed spline forms
  // a continuous loop: the first and last points are the same, and
  // derivatives are continuous.

  int Closed;

  // Explicitly specify the parametric range.
  double ParametricRange[2];

  // Helper methods
  double ComputeLeftDerivative();
  double ComputeRightDerivative();
  int FindIndex(int size, double t);

private:
  Spline(const Spline&);  // Not implemented.
  void operator=(const Spline&);  // Not implemented.
};

#endif
