/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "turbine_hover.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "util/loading/factory.h"
#include "key_mapper.h"
#include "state.h"

#include "graphics_engine.h"
#include "particles/dot_emitter.h"
#include "particles/sprite_particles.h"

#include "debug.h"


ObjectListDefinition(TurbineHover);
CREATE_FACTORY(TurbineHover);

/**
 *  destructs the turbine_hover, deletes alocated memory
 */
TurbineHover::~TurbineHover ()
{
  this->setPlayer(NULL);
}

/**
 * @brief loads a TurbineHover information from a specified file.
 * @param fileName the name of the File to load the turbine_hover from (absolute path)
 */
TurbineHover::TurbineHover(const std::string& fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for TurbineHover.\n", fileName.c_str());
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 * @brief creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
TurbineHover::TurbineHover(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  //Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_HYPERBLASTER));

  //  cannon->setName("BFG");

  this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1 ,1);
  //this->addWeapon(cannon, 0, 2);

  this->getWeaponManager().changeWeaponConfig(1);
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

  this->loadModel("models/ships/hoverglider_mainbody.obj");

}


/**
 * @brief initializes a TurbineHover
 */
void TurbineHover::init()
{
  //  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->registerObject(this, TurbineHover::_objectList);

  this->setSupportedPlaymodes(Playable::Full3D | Playable::Horizontal | Playable::Vertical);

  this->loadModel("models/ships/hoverglider_wing.obj", 1.0f, 3);
  this->loadModel("models/ships/hoverglider_turbine.obj", 1.0f, 4);
  this->loadModel("models/ships/hoverglider_turbine_rotors.obj", 1.0f, 5);

  bForward = bBackward = bLeft = bRight = bAscend = bDescend = false;
  mouseSensitivity = 0.005;

  this->rotorSpeed = 1000.0f;
  this->rotorCycle = 0.0f;
  this->cameraLook = 0.0f;
  this->rotation = 0.0f;
  this->acceleration = 10.0f;
  this->airFriction = 2.0f;

  this->setHealthMax(100);
  this->setHealth(100);


  // camera - issue
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  //this->cameraNode.setParentMode(PNODE_ROTATE_MOVEMENT);
  this->cameraNode.setParent(this);
  this->cameraNode.setRelCoor(0,5,0);

  // rotors
  this->wingNodeLeft.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT );
  this->wingNodeLeft.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  this->wingNodeLeft.setParent(this);
  this->wingNodeLeft.setRelCoor(-1.5, -.3, -1.0);
  this->rotorNodeLeft.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->rotorNodeLeft.setParent(&this->wingNodeLeft);
  this->rotorNodeLeft.setRelCoor(0, 1.0, -2.3);

  this->wingNodeRight.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->wingNodeRight.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  this->wingNodeRight.setParent(this);
  this->wingNodeRight.setRelCoor(-1.5, -0.3, 1.0);
  this->rotorNodeRight.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->rotorNodeRight.setParent(&this->wingNodeRight);
  this->rotorNodeRight.setRelCoor(0, 1.0, 2.3);

  // PARTICLES
  this->burstEmitter[0] = new DotEmitter(200, 5.0, .01);
  this->burstEmitter[0]->setParent(&this->rotorNodeLeft);
  this->burstEmitter[0]->setRelCoor(0, -0.7, 0);
  this->burstEmitter[0]->setRelDir(Quaternion(-M_PI_2, Vector(0,0,1)));
  this->burstEmitter[0]->setName("TurbineHover_Burst_emitter_Left");

  this->burstEmitter[1] = new DotEmitter(200, 5.0, .01);
  this->burstEmitter[1]->setParent(&this->rotorNodeRight);
  this->burstEmitter[1]->setRelCoor(0, -0.7, 0);
  this->burstEmitter[1]->setRelDir(Quaternion(-M_PI_2, Vector(0,0,1)));
  this->burstEmitter[1]->setName("TurbineHover_Burst_emitter_Right");


  this->burstSystem = new SpriteParticles(1000);
  this->burstSystem->addEmitter(this->burstEmitter[0]);
  this->burstSystem->addEmitter(this->burstEmitter[1]);
  this->burstSystem->setName("SpaceShip_Burst_System");
  ((SpriteParticles*)this->burstSystem)->setMaterialTexture("textures/radial-trans-noise.png");
  this->burstSystem->setLifeSpan(1.0, .3);
  this->burstSystem->setRadius(0.0, 1.5);
  this->burstSystem->setRadius(0.05, 1.8);
  this->burstSystem->setRadius(.5, .8);
  this->burstSystem->setRadius(1.0, 0);
  this->burstSystem->setColor(0.0, .7,.7,1,.5);
  this->burstSystem->setColor(0.2, 0,0,0.8,.5);
  this->burstSystem->setColor(0.5, .5,.5,.8,.3);
  this->burstSystem->setColor(1.0, .8,.8,.8,.0);


  //add events to the eventlist of the Playable
  this->registerEvent(KeyMapper::PEV_FORWARD);
  this->registerEvent(KeyMapper::PEV_BACKWARD);
  this->registerEvent(KeyMapper::PEV_LEFT);
  this->registerEvent(KeyMapper::PEV_RIGHT);
  this->registerEvent(KeyMapper::PEV_UP);
  this->registerEvent(KeyMapper::PEV_DOWN);
  this->registerEvent(KeyMapper::PEV_FIRE1);
  this->registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  this->registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  this->registerEvent(EV_MOUSE_MOTION);

  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

  // WEAPON_MANAGER configuration
  this->getWeaponManager().setSlotCount(5);

  this->getWeaponManager().setSlotPosition(0, Vector(-0.28, 1.186, -2.750), &this->wingNodeLeft);
  this->getWeaponManager().setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager().setSlotPosition(1, Vector(-0.28, 1.186, 2.750), &this->wingNodeRight);
  this->getWeaponManager().setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager().setSlotPosition(2, Vector(-1.63, .809, -.003));
  this->getWeaponManager().setSlotCapability(2, WTYPE_HEAVY);

  /// TODO: THESE ARE TOO MUCH
  this->getWeaponManager().setSlotPosition(3, Vector(-1.63, .678, -.652));
  this->getWeaponManager().setSlotDirection(3, Quaternion(-24/180 * M_PI, Vector(1,0,0)));

  this->getWeaponManager().setSlotPosition(4, Vector(-1.63, .678, .652));
  this->getWeaponManager().setSlotDirection(4, Quaternion(24/180 * M_PI, Vector(1,0,0)));

  this->cameraNode.setRelCoor(1,5,0);
  this->getWeaponManager().getFixedTarget()->setParent(&this->cameraNode);
  this->getWeaponManager().getFixedTarget()->setRelCoor(1000,0,0);

  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bAscend, &bAscend, "bAscend", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bDescend, &bDescend, "bDescend", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableQuaternion( &direction, &direction, "direction", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &cameraLook, &cameraLook, "cameraLook", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &rotation, &rotation, "rotation", PERMISSION_OWNER ) );

  if( State::isOnline())
    toList( OM_PLAYERS );
}

/**
 * @brief loads the Settings of a TurbineHover from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void TurbineHover::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);
}


void TurbineHover::setBoostColor(const Color& color)
{
  this->burstSystem->setColor(0.0, color);
  this->burstSystem->setColor(0.2, color * 0.6);
  this->burstSystem->setColor(0.5, color * .3 + Color(0.5, 0.5, 0.8, 0.3));
  this->burstSystem->setColor(1.0, 0.8, 0.8, 0.8, 0.0);

  printf(":::::::::::::::::::::::\n");
  this->burstSystem->debug();
}

void TurbineHover::setPlayDirection(const Quaternion& rot, float speed)
{
  this->direction = Quaternion (rot.getHeading(), Vector(0,1,0));
}

void TurbineHover::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( true);

  if (State::getCameraNode != NULL)
  {
    State::getCameraNode()->setParentSoft(&this->cameraNode);
    State::getCameraNode()->setRelCoorSoft(-10, 0,0);
    State::getCameraTargetNode()->setParentSoft(&this->cameraNode);
  }
}

void TurbineHover::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
  this->detachCamera();

}

void TurbineHover::setTeam(int teamID)
{
  printf("::::::::::::::::: TEAM ::: %d\n", teamID);
  if (teamID == 0)
  {
    this->setBoostColor(Color::blue);
  }
  else if (teamID == 1)
  {
    this->setBoostColor(Color::red);
  }

  ///HACK this is very much hard coded.set the owner of the weapons
  this->getWeaponManager().getWeapon(0)->setOwner(this->getOwner());
  this->getWeaponManager().getWeapon(1)->setOwner(this->getOwner());

  //choose collision list
  if( State::isOnline())
  {
    if( teamID == 0)
      toList( OM_GROUP_00 );
    else if(teamID == 1)
      toList( OM_GROUP_01);
  }


  // set the local team id, very important
  this->setTeamId(teamID);
}



/**
 * @brief effect that occurs after the TurbineHover is spawned
*/
void TurbineHover::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 * @brief the action occuring if the turbine_hover left the game
*/
void TurbineHover::leftWorld ()
{}

/**
 * @brief this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void TurbineHover::collidesWith(WorldEntity* entity, const Vector& location)
{
  Playable::collidesWith(entity, location);
}



/**
 * @brief the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void TurbineHover::tick (float dt)
{
  //  this->debugNode(1);
  Playable::tick(dt);

  // spaceship controlled movement

  // TRYING TO FIX PNode.
  //this->cameraNode.setRelCoor(Vector(0.0f, 5.0f, 0.0f));//, 30.0f);
  //this->cameraNode.setRelDir(Quaternion(this->smoothRotator  *10.0, Vector(0,1,0)));

  this->movement(dt);
  this->rotorCycle += this->rotorSpeed * dt;
}


#include "shared_network_data.h"

/**
 * @brief calculate the velocity
 * @param time the timeslice since the last frame
*/
void TurbineHover::movement (float dt)
{
  Vector accel(0.0, 0.0, 0.0);

  if( this->bForward )
  {
    accel += Vector(this->acceleration, 0, 0);
  }

  if( this->bBackward )
  {
    accel -= Vector(this->acceleration, 0, 0);
  }
  if( this->bLeft)
  {
    accel -= Vector(0, 0, this->acceleration);
  }

  if( this->bRight)
  {
    accel += Vector(0, 0, this->acceleration);
  }

  if (this->bAscend )
  {
    accel += Vector(0, this->acceleration, 0);
  }
  if (this->bDescend )
  {
    accel -= Vector(0, this->acceleration, 0);
  }

  switch(this->getPlaymode())
  {
    case Playable::Full3D:
      {
        Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);

        // this is the air friction (necessary for a smooth control)
        Vector damping = (this->velocity * this->airFriction);


        this->velocity += (accelerationDir - damping)* dt;
        //if (this->getOwner() != SharedNetworkData::getInstance()->getHostID())
          this->shiftCoor (this->velocity * dt);

        // limit the maximum rotation speed.
        if (this->rotation != 0.0f)
        {
          float maxRot = 10.0 * dt;
          if (unlikely(this->rotation > maxRot)) this->rotation = maxRot;
          if (unlikely(this->rotation < -maxRot)) this->rotation = -maxRot;
          this->direction *= Quaternion(-M_PI/4.0*this->rotation, Vector(0,1,0));

          this->rotation = 0.0f;
        }

        //if (this->getOwner() != SharedNetworkData::getInstance()->getHostID())
          this->setRelDirSoft(this->direction * Quaternion(-cameraLook, Vector(0,0,1)), 5);

        this->wingNodeLeft.setRelDirSoft(Quaternion(accel.z * .05 +this->rotation, Vector(1,0,0)), 5);
        this->rotorNodeLeft.setRelDirSoft(Quaternion(-accel.x * .07+this->rotation + cameraLook, Vector(0,0,1)), 5);

        this->wingNodeRight.setRelDirSoft(Quaternion(accel.z * .05 +this->rotation, Vector(1,0,0)), 5);
        this->rotorNodeRight.setRelDirSoft(Quaternion(-accel.x*.07 -this->rotation + cameraLook, Vector(0,0,1)), 5);
      }
      break;

    case Playable::Horizontal:
      {
        accel.y = 0.0;
        Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);
        accelerationDir.y = 0.0;

        // this is the air friction (necessary for a smooth control)
        Vector damping = (this->velocity * this->airFriction);


        this->velocity += (accelerationDir - damping)* dt;
        this->shiftCoor (this->velocity * dt);

        // limit the maximum rotation speed.
        if (this->rotation != 0.0f)
        {
          float maxRot = 10.0 * dt;
          if (unlikely(this->rotation > maxRot)) this->rotation = maxRot;
          if (unlikely(this->rotation < -maxRot)) this->rotation = -maxRot;
          this->direction *= Quaternion(-M_PI/4.0*this->rotation, Vector(0,1,0));

          this->rotation = 0.0f;
        }

        this->setRelDirSoft(this->direction, 5);

        this->wingNodeLeft.setRelDirSoft(Quaternion(accel.z * .05 +this->rotation, Vector(1,0,0)), 5);
        this->rotorNodeLeft.setRelDirSoft(Quaternion(-accel.x * .07+this->rotation, Vector(0,0,1)), 5);

        this->wingNodeRight.setRelDirSoft(Quaternion(accel.z * .05 +this->rotation, Vector(1,0,0)), 5);
        this->rotorNodeRight.setRelDirSoft(Quaternion(-accel.x*.07 -this->rotation, Vector(0,0,1)), 5);
      }
      break;

    case Playable::Vertical:
      {
        accel.z = 0;
        Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);
        accelerationDir.z=0;

        // this is the air friction (necessary for a smooth control)
        Vector damping = (this->velocity * this->airFriction);

        this->velocity += (accelerationDir - damping)* dt;
        this->shiftCoor (this->velocity * dt);

        this->setRelDirSoft(this->direction * Quaternion(-cameraLook, Vector(0,0,1)), 5);

        this->wingNodeLeft.setRelDirSoft(Quaternion(accel.z * .05 +this->rotation, Vector(1,0,0)), 5);
        this->rotorNodeLeft.setRelDirSoft(Quaternion(-accel.x * .07+this->rotation + cameraLook, Vector(0,0,1)), 5);

        this->wingNodeRight.setRelDirSoft(Quaternion(accel.z * .05 +this->rotation, Vector(1,0,0)), 5);
        this->rotorNodeRight.setRelDirSoft(Quaternion(-accel.x*.07 -this->rotation + cameraLook, Vector(0,0,1)), 5);
      }
      break;
    default:
      PRINTF(2)("Playmode %s Not Implemented\n", Playable::playmodeToString(this->getPlaymode()).c_str());
  }
}


void TurbineHover::draw() const
{
  Vector tmpRot;
  WorldEntity::draw();

  glPushMatrix();
  /// LEFT SIDE
  glTranslatef (this->wingNodeLeft.getAbsCoor ().x,
                this->wingNodeLeft.getAbsCoor ().y,
                this->wingNodeLeft.getAbsCoor ().z);
  tmpRot = this->wingNodeLeft.getAbsDir().getSpacialAxis();
  glRotatef (this->wingNodeLeft.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->getModel(3)->draw();
  glPopMatrix ();

  glPushMatrix();
  glTranslatef (this->rotorNodeLeft.getAbsCoor ().x,
                this->rotorNodeLeft.getAbsCoor ().y,
                this->rotorNodeLeft.getAbsCoor ().z);
  tmpRot = this->rotorNodeLeft.getAbsDir().getSpacialAxis();
  glRotatef (this->rotorNodeLeft.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->getModel(4)->draw();
  glRotatef(this->rotorCycle, 0,1,0);
  this->getModel(5)->draw();
  glPopMatrix ();

  /// RIGHT SIDE
  glPushMatrix();
  glTranslatef (this->wingNodeRight.getAbsCoor ().x,
                this->wingNodeRight.getAbsCoor ().y,
                this->wingNodeRight.getAbsCoor ().z);
  tmpRot = this->wingNodeRight.getAbsDir().getSpacialAxis();
  glRotatef (this->wingNodeRight.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  glScalef(1,1,-1);
  this->getModel(3)->draw();
  glPopMatrix ();

  glPushMatrix();
  glTranslatef (this->rotorNodeRight.getAbsCoor ().x,
                this->rotorNodeRight.getAbsCoor ().y,
                this->rotorNodeRight.getAbsCoor ().z);
  tmpRot = this->rotorNodeRight.getAbsDir().getSpacialAxis();
  glRotatef (this->rotorNodeRight.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  glScalef(1,1,-1);
  this->getModel(4)->draw();
  glRotatef(this->rotorCycle, 0,1,0);
  this->getModel(5)->draw();
  glPopMatrix ();
}

/**
 * @todo switch statement ??
 */
void TurbineHover::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
    this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_UP)
    this->bAscend = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_DOWN)
    this->bDescend = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == KeyMapper::PEV_FORWARD)
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_BACKWARD)
    this->bBackward = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    float xMouse, yMouse;
    xMouse = event.xRel*mouseSensitivity;
    yMouse = event.yRel*mouseSensitivity;

    // rotate the Player around the y-axis
    this->rotation += xMouse;

    this->cameraLook += yMouse;
    // rotate the Camera around the z-axis
    if (cameraLook > M_PI_4)
      cameraLook = M_PI_4;
    else if (cameraLook < -M_PI_4)
      cameraLook = -M_PI_4;
    //this->cameraNode.setRelDirSoft(this->direction,10);
  }
}


/**
 *  respawning function called by the GameRules
 */
void TurbineHover::respawn( )
{

  Playable::respawn();
}
