/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "spacecraft_2d.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "util/loading/factory.h"
#include "util/loading/load_param.h"
#include "key_mapper.h"
#include "state.h"

#include "graphics_engine.h"
#include "particles/dot_emitter.h"
#include "particles/sprite_particles.h"

#include "debug.h"

#include "script_class.h"



ObjectListDefinition(Spacecraft2D);
CREATE_FACTORY(Spacecraft2D);

CREATE_SCRIPTABLE_CLASS(Spacecraft2D,
			addMethod("hasPlayer", Executor0ret<Playable, lua_State*,bool>(&Playable::hasPlayer))
                        //Coordinates
			    ->addMethod("setAbsCoor", Executor3<PNode, lua_State*,float,float,float>(&PNode::setAbsCoor))
			    ->addMethod("getAbsCoorX", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorX))
			    ->addMethod("getAbsCoorY", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorY))
			    ->addMethod("getAbsCoorZ", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorZ))
			    ->addMethod("setAirFriction", Executor1<Spacecraft2D, lua_State*, float>(&Spacecraft2D::setAirFriction))
                       );



/**
 * @brief loads a Spacecraft2D information from a specified file.
 * @param fileName the name of the File to load the spacecraft_2d from (absolute path)
 */
Spacecraft2D::Spacecraft2D(const std::string& fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for Spacecraft2D.\n", fileName.c_str());
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 * @brief creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
Spacecraft2D::Spacecraft2D(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);



  //weapons:
  Weapon* wpRight = dynamic_cast<Weapon*>(Factory::fabricate("LaserCannon"));
  wpRight->setName("Cannon_Right");
  Weapon* wpLeft = dynamic_cast<Weapon*>(Factory::fabricate("LaserCannon"));
  wpLeft->setName("Cannon_Left");

  Weapon* turretLeft = dynamic_cast<Weapon*>(Factory::fabricate("BoomerangGun"));
  wpRight->setName("Turret_Left");
  Weapon* turretRight = dynamic_cast<Weapon*>(Factory::fabricate("BoomerangGun"));
  wpLeft->setName("Turret_Right");

  //  cannon->setName("BFG");

  this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1 ,1);
  this->addWeapon(turretLeft, 1, 2);
  this->addWeapon(turretRight, 1, 3);

  //this->addWeapon(cannon, 0, 2);

  this->getWeaponManager().changeWeaponConfig(1);
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
}


/**
 * @brief destructs the spacecraft_2d, deletes alocated memory
 */
Spacecraft2D::~Spacecraft2D ()
{
  this->setPlayer(NULL);
  delete this->toTravelHeight;
}


/**
 * @brief initializes a Spacecraft2D
 */
void Spacecraft2D::init()
{
  //  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->registerObject(this, Spacecraft2D::_objectList);

  this->setSupportedPlaymodes(Playable::Full3D | Playable::Horizontal );

  bForward = bBackward = bLeft = bRight = false;
  mouseSensitivity = 0.005;

  this->cameraLook = 0.0f;
  this->rotation = 0.0f;
  this->acceleration = 20.0f;
  this->airFriction = 0.0f;


  this->setHealthMax(1000);
  this->setHealth(1000);
  this->setDamage(100.0f);



  /// 2D-MODE
  this->toTravelHeight = NULL;
  this->travelSpeed = 0.0f;
  this->travelNode = new PNode();

  this->loadModel("models/ships/mantawing.obj", 5.0f);

  // camera - issue
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  //this->cameraNode.setParentMode(PNODE_ROTATE_MOVEMENT);
  //this->cameraNode.setParent(this);

  // PARTICLES
  this->burstEmitter = new DotEmitter(200, 5.0, .01);
  this->burstEmitter->setParent(this);
  this->burstEmitter->setRelCoor(0, -0.7, 0);
  this->burstEmitter->setRelDir(Quaternion(-M_PI, Vector(0,0,1)));
  this->burstEmitter->setName("Spacecraft2D_Burst_emitter_Left");

  this->burstSystem = new SpriteParticles(1000);
  this->burstSystem->addEmitter(this->burstEmitter);
  this->burstSystem->setName("SpaceShip_Burst_System");
  ((SpriteParticles*)this->burstSystem)->setMaterialTexture("textures/radial-trans-noise.png");
  this->burstSystem->setLifeSpan(1.0, .3);
  this->burstSystem->setRadius(0.0, 1.5);
  this->burstSystem->setRadius(0.05, 1.8);
  this->burstSystem->setRadius(.5, .8);
  this->burstSystem->setRadius(1.0, 0);
  this->burstSystem->setColor(0.0, .7,.7,1,.5);
  this->burstSystem->setColor(0.2, 0,0,0.8,.5);
  this->burstSystem->setColor(0.5, .5,.5,.8,.3);
  this->burstSystem->setColor(1.0, .8,.8,.8,.0);


  //add events to the eventlist of the Playable
  this->registerEvent(KeyMapper::PEV_FORWARD);
  this->registerEvent(KeyMapper::PEV_BACKWARD);
  this->registerEvent(KeyMapper::PEV_LEFT);
  this->registerEvent(KeyMapper::PEV_RIGHT);
  this->registerEvent(KeyMapper::PEV_FIRE1);
  this->registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  this->registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  this->registerEvent(EV_MOUSE_MOTION);

  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);

  // WEAPON_MANAGER configuration
  this->getWeaponManager().setSlotCount(5);

  this->getWeaponManager().setSlotPosition(0, Vector(1.843, -0.335, 2.029) * 5.0);
  this->getWeaponManager().setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager().setSlotPosition(1, Vector(1.843, -0.335, -2.029) * 5.0);
  this->getWeaponManager().setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  /// TODO: THESE ARE TOO MUCH
  this->getWeaponManager().setSlotPosition(2, Vector(-0.351, -.238, 1.406) * 5.0);
  this->getWeaponManager().setSlotDirection(2, Quaternion(-1.7, Vector(0,1,0)));

  this->getWeaponManager().setSlotPosition(3, Vector(-0.351, -.238, -1.406) * 5.0);
  this->getWeaponManager().setSlotDirection(3, Quaternion(1.7, Vector(0,1,0)));

  this->cameraNode.setRelCoor(1,5,0);
  this->getWeaponManager().getFixedTarget()->setParent(&this->cameraNode);
  this->getWeaponManager().getFixedTarget()->setRelCoor(1000,0,0);

  registerVar( new SynchronizeableBool( &bForward, &bForward, "bForward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bBackward, &bBackward, "bBackward", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bLeft, &bLeft, "bLeft", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableBool( &bRight, &bRight, "bRight", PERMISSION_OWNER ) );
  //registerVar( new SynchronizeableQuaternion( &direction, &direction, "direction", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &cameraLook, &cameraLook, "cameraLook", PERMISSION_OWNER ) );
  registerVar( new SynchronizeableFloat( &rotation, &rotation, "rotation", PERMISSION_OWNER ) );

  this->setDamage( 1000.0f);
  this->subscribeReaction(CoRe::CREngine::CR_OBJECT_DAMAGE, WorldEntity::staticClassID());

}

/**
 * @brief loads the Settings of a Spacecraft2D from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void Spacecraft2D::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);

  LoadParam(root, "travel-speed", this, Spacecraft2D, setTravelSpeed);
  LoadParam(root, "travel-height", this, Spacecraft2D, setTravelHeight);
  LoadParam(root, "travel-distance", this, Spacecraft2D, setTravelDistance);
}

void Spacecraft2D::setPlayDirection(const Quaternion& rot, float speed)
{
  this->direction = Quaternion (rot.getHeading(), Vector(0,1,0));
}

void Spacecraft2D::setTravelSpeed(float travelSpeed)
{
  this->travelSpeed = travelSpeed;
}


void Spacecraft2D::setTravelHeight(float travelHeight)
{
  if (this->toTravelHeight == NULL)
    this->toTravelHeight = new float;
  *this->toTravelHeight = travelHeight;
}


void Spacecraft2D::setTravelDistance(const Vector2D& distance)
{
  this->travelDistance = distance;
}

void Spacecraft2D::setTravelDistance(float x, float y)
{
  this->setTravelDistance(Vector2D(x, y));
}



void Spacecraft2D::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( true);
  this->setPlaymode(this->getPlaymode());
}

void Spacecraft2D::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager().getFixedTarget())->setVisibility( false);
  this->detachCamera();

}


void Spacecraft2D::enterPlaymode(Playable::Playmode playmode)
{
  switch(playmode)
  {
    case Playable::Full3D:
      if (State::getCameraNode != NULL)
      {
        Vector absCoor = this->getAbsCoor();
        this->setParent(PNode::getNullParent());
        this->setAbsCoor(absCoor);
        State::getCameraNode()->setParentSoft(&this->cameraNode);
        State::getCameraNode()->setRelCoorSoft(-10, 0,0);
        State::getCameraTargetNode()->setParentSoft(&this->cameraNode);
        State::getCameraTargetNode()->setRelCoorSoft(100, 0,0);

      }
      break;


    case Playable::Horizontal:
      if (State::getCameraNode != NULL)
      {
        this->debugNode(1);
        this->travelNode->debugNode(1);

        this->travelNode->setAbsCoor(this->getAbsCoor());
        this->travelNode->updateNode(0.01f);

        this->setParent(this->travelNode);
        this->setRelCoor(0,0,0);

        State::getCameraNode()->setParentSoft(this->travelNode);
        State::getCameraNode()->setRelCoorSoft(-3, 100,0);
        State::getCameraTargetNode()->setParentSoft(this->travelNode);
        State::getCameraTargetNode()->setRelCoorSoft(5,0,1);


        this->debugNode(1);
        this->travelNode->debugNode(1);
      }
      break;

    default:
      PRINTF(2)("Playmode %s Not Implemented in %s\n", Playable::playmodeToString(this->getPlaymode()).c_str(), this->getClassCName());
  }
}



/**
 * @brief effect that occurs after the Spacecraft2D is spawned
*/
void Spacecraft2D::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 * @brief the action occuring if the spacecraft_2d left the game
*/
void Spacecraft2D::leftWorld ()
{}

/**
 * @brief this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void Spacecraft2D::collidesWith(WorldEntity* entity, const Vector& location)
{
  Playable::collidesWith(entity, location);
}



/**
 * @brief the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void Spacecraft2D::tick (float dt)
{
  //  this->debugNode(1);
  Playable::tick(dt);

  // spaceship controlled movement
  this->movement(dt);

  // TRYING TO FIX PNode.
  this->cameraNode.setAbsCoorSoft(this->getAbsCoor() + Vector(0.0f, 5.0f, 0.0f), 30.0f);
  this->cameraNode.setRelDirSoft(this->getAbsDir(), 30.0f);
}

/**
 * @brief calculate the velocity
 * @param time the timeslice since the last frame
*/
void Spacecraft2D::movement (float dt)
{
  Vector accel(0.0, 0.0, 0.0);

  if( this->bForward )
  {
    accel += Vector(this->acceleration, 0, 0);
  }

  if( this->bBackward )
  {
    accel -= Vector(this->acceleration, 0, 0);
  }
  if( this->bLeft)
  {
    accel -= Vector(0, 0, this->acceleration);
  }

  if( this->bRight)
  {
    accel += Vector(0, 0, this->acceleration);
  }

  switch(this->getPlaymode())
  {
    case Playable::Full3D:
      {
        Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);

        // this is the air friction (necessary for a smooth control)
        Vector damping = (this->velocity * this->airFriction);


        this->velocity += (accelerationDir - damping)* dt;
        this->shiftCoor (this->velocity * dt);

        // limit the maximum rotation speed.
        if (this->rotation != 0.0f)
        {
          float maxRot = 10.0 * dt;
          if (unlikely(this->rotation > maxRot)) this->rotation = maxRot;
          if (unlikely(this->rotation < -maxRot)) this->rotation = -maxRot;
          this->direction *= Quaternion(-M_PI/4.0*this->rotation, Vector(0,1,0));

          this->rotation = 0.0f;
        }

        this->setRelDirSoft(this->direction * Quaternion(-cameraLook, Vector(0,0,1)), 5);
      }
      break;

    case Playable::Horizontal:
      {

        if (this->toTravelHeight != NULL)
        {
          this->travelNode->shiftCoor(Vector(0, (*toTravelHeight - this->travelNode->getAbsCoor().y) * dt * 10.0, 0));
          if (fabsf(this->travelNode->getAbsCoor().y - *this->toTravelHeight) < .1)
          {
            delete this->toTravelHeight;
            this->toTravelHeight = NULL;
          }
        }
        this->travelNode->shiftCoor(Vector(this->travelSpeed * dt, 0, 0));

        accel.y = 0.0;

        Vector accelerationDir = this->getAbsDir().apply(accel * this->acceleration);
        accelerationDir.y = 0.0;

        // this is the air friction (necessary for a smooth control)
        Vector damping = (this->velocity * this->airFriction);


        this->velocity += (accelerationDir - damping)* dt;

        if (this->getRelCoor().z > this->travelDistance.y && velocity.z > 0.0)
          this->velocity.z = 0.0f;
        if (this->getRelCoor().z < -this->travelDistance.y && velocity.z < 0.0)
          this->velocity.z = 0.0f;

        if (this->getRelCoor().x > this->travelDistance.x && velocity.x > 0.0)
          this->velocity.x = 0.0f;
        if (this->getRelCoor().x < -this->travelDistance.x && velocity.x < 0.0)
          this->velocity.x = 0.0f;


        this->shiftCoor (this->velocity * dt);
        if (accel.z == 0)
          this->setRelDirSoft(Quaternion(0, Vector(0,0,0)), 5.0f);
        else
          this->setRelDirSoft(Quaternion(this->velocity.z * .004, Vector(1,0,0)), 4.5f);
      }
      break;

    default:
      PRINTF(2)("Playmode %s Not Implemented in %s\n", Playable::playmodeToString(this->getPlaymode()).c_str(), this->getClassCName());
  }
}


void Spacecraft2D::draw() const
{
  WorldEntity::draw();
}

/**
 * @todo switch statement ??
 */
void Spacecraft2D::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
    this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
    this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_FORWARD)
    this->bForward = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_BACKWARD)
  {this->bBackward = event.bPressed; printf(" %f, %f, %f \n",getAbsCoorX(),getAbsCoorY(),getAbsCoorZ());} //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {



    if (this->getPlaymode() == Playable::Full3D)
    {
      float xMouse, yMouse;
      xMouse = event.xRel*mouseSensitivity;
      yMouse = event.yRel*mouseSensitivity;

      // rotate the Player around the y-axis
      this->rotation += xMouse;

      this->cameraLook += yMouse;
      // rotate the Camera around the z-axis
      if (cameraLook > M_PI_4)
        cameraLook = M_PI_4;
      else if (cameraLook < -M_PI_4)
        cameraLook = -M_PI_4;
    }
  }
}
