/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "sprite_particles.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "material.h"
#include "state.h"
#include "shell_command.h"



ObjectListDefinition(SpriteParticles);
CREATE_FACTORY(SpriteParticles);

SHELL_COMMAND(texture, SpriteParticles, setMaterialTexture)
    ->defaultValues("textures/evil-flower.png");



/**
 *  standard constructor
 * @param maxCount the Count of particles in the System
 * @param type The Type of the SpriteParticles
*/
SpriteParticles::SpriteParticles (unsigned int maxCount)
  : ParticleSystem(maxCount)
{
  this->init();
}

/**
 * @brief creates a Particle System out of a XML-element
 * @param root: the XML-element to load from
 */
SpriteParticles::SpriteParticles(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);
}

/**
 *  standard deconstructor
*/
SpriteParticles::~SpriteParticles()
{ }

/**
 * @brief initializes the SpriteParticles with default values
*/
void SpriteParticles::init()
{
  this->registerObject(this, SpriteParticles::_objectList);

  this->material.setDiffuseMap("textures/radial-trans-noise.png");
}


/**
 * loads Parameters from a TiXmlElement
 * @param root the XML-element to load from.
 */
void SpriteParticles::loadParams(const TiXmlElement* root)
{
  ParticleSystem::loadParams(root);

  LoadParam(root, "texture", this, SpriteParticles, setMaterialTexture);
}

/**
 * @brief sets the Texutre that is placed onto the particles
 * @param textureFile the Texture to load onto these SpriteParticles
 */
void SpriteParticles::setMaterialTexture(const std::string& textureFile)
{
  this->material.setDiffuseMap(textureFile);
}

/**
 * @brief draws all the Particles of this System
 *
 * The Cases in this Function all do the same:
 * Drawing all the particles with the appropriate Type.
 * This is just the fastest Way to do this, but will most likely be changed in the future.
 */
void SpriteParticles::draw() const
{

  Particle* drawPart = particles;
  this->material.select();

  GLboolean checkLight = false;
  glGetBooleanv(GL_LIGHTING, &checkLight);
  glPushAttrib(GL_LIGHTING_BIT);
  if (checkLight == GL_TRUE)
    glDisable(GL_LIGHTING);

  glDepthMask(GL_FALSE);
  /*
  glClearDepth(1.0);
  glDepthFunc(GL_LEQUAL);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glAlphaFunc(GL_GREATER,0.1);
  glEnable(GL_ALPHA_TEST);
  glEnable(GL_TEXTURE_2D);
  glEnable(GL_CULL_FACE);
  */
  //glDepthMask(GL_FALSE);
//glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_ENV_MODE, GL_MODULATE);


  while (likely(drawPart != NULL))
  {
    glColor4fv(drawPart->color);
    //! @todo implement a faster code for the look-at Camera algorithm.

    const PNode* camera = State::getCameraNode();  //!< @todo MUST be different
    Vector cameraPos = camera->getAbsCoor();
    Vector cameraTargetPos = State::getCameraTargetNode()->getAbsCoor();
    Vector view = cameraTargetPos - cameraPos;
    Vector up = Vector(0, 1, 0);
    up = camera->getAbsDir().apply(up);
    Vector h = up.cross(view);
    Vector v = h.cross(view);
    h.normalize();
    v.normalize();
    v *= .5 * drawPart->radius;
    h *= .5 * drawPart->radius;

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2i(1, 1);
    glVertex3f(drawPart->position.x - h.x - v.x,
               drawPart->position.y - h.y - v.y,
               drawPart->position.z - h.z - v.z);
    glTexCoord2i(0, 1);
    glVertex3f(drawPart->position.x - h.x + v.x,
               drawPart->position.y - h.y + v.y,
               drawPart->position.z - h.z + v.z);
    glTexCoord2i(1, 0);
    glVertex3f(drawPart->position.x + h.x - v.x,
               drawPart->position.y + h.y - v.y,
               drawPart->position.z + h.z - v.z);
    glTexCoord2i(0, 0);
    glVertex3f(drawPart->position.x + h.x + v.x,
               drawPart->position.y + h.y + v.y,
               drawPart->position.z + h.z + v.z);

    glEnd();

    drawPart = drawPart->next;
  }
  glDepthMask(GL_TRUE);
  glPopAttrib();
}
