

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Patrick Boenzli
   co-programmer:
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "util/loading/factory.h"
#include "util/loading/load_param.h"

#include "interactive_model.h"


#include "repair_station.h"


ObjectListDefinition(RepairStation);
CREATE_FACTORY(RepairStation);



//! list of all different animations a std md2model supports
// sAnim RepairStation::animationList[2] =
// {
//  // begin, end, fps, interruptable
//   {   0,  10,  30,  0 },   //!< OPEN
//   {   10, 20,  30,  0 }    //!< CLOSE
// };


sAnim RepairStation::animationList[8] =
{
 // begin, end, fps, interruptable
  {   0,  12,  20,  0 },   //!< CYCLE01
  {   12, 24,  30,  0 },   //!< CYCLE02
  {   24, 40,  10,  0 },   //!< CYCLE03
  {   40, 55,  30,  0 },   //!< CYCLE04
  {   55, 68,  20,  0 },   //!< CYCLE05
  {   68, 81,  30,  0 },   //!< CYCLE06
  {   81, 89,  40,  0 },   //!< CYCLE07
  {   89, 99,  30,  0 }    //!< CYCLE08
};



RepairStation::RepairStation ()
{
  this->init();
}


RepairStation::RepairStation(const TiXmlElement* root)
{
  this->registerObject(this, RepairStation::_objectList);
  this->scale = 1.0f;

  if( root != NULL)
    this->loadParams(root);

  this->toList(OM_COMMON);

  this->bActivated = true;
  this->animationStep = 1;
  this->animationCurrent = REPAIR_CYCLE01;

  this->loadMD2Texture("textures/repairstation.jpg");
  this->loadModel("models/creatures/repairstation.md2", this->scale);

  this->setAnimation(REPAIR_CYCLE01, MD2_ANIM_ONCE);
}


RepairStation::~RepairStation ()
{}



/**
 * loads the Settings of a MD2Creature from an XML-element.
 * @param root the XML-element to load the MD2Creature's properties from
 */
void RepairStation::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "scale", this, RepairStation, setScale)
      .describe("sets the scale of the repair station")
      .defaultValues(1.0);
}


/**
 * sets the animatin of this entity
 */
void  RepairStation::setAnimation(int animNum, int playbackMode)
{
  if( likely(this->getModel(0) != NULL))
    ((InteractiveModel*)this->getModel(0))->setAnimation(animationList[animNum].firstFrame,
                                                         animationList[animNum].lastFrame,
                                                         animationList[animNum].fps,
                                                         animationList[animNum].bStoppable,
                                                         playbackMode);
}


/**
 * @returns the current animation number
 */
int RepairStation::getAnimation()
{
  if( likely(this->getModel(0) != NULL))
    return ((InteractiveModel*)this->getModel(0))->getAnimation();
  else
    return -1;
}



/**
 * @returns true if animation is finished
 */
bool RepairStation::isAnimationFinished()
{
  if( likely(this->getModel(0) != NULL))
    return ((InteractiveModel*)this->getModel(0))->isAnimationFinished();
  else
    return false;
}


/**
 * this activates the repair station and makes it working
 */
void RepairStation::activate()
{
  this->animationStep = 1;
  this->bActivated = true;
}


/**
 * this deactivates the repair station
 */
void RepairStation::deactivate()
{
  this->animationStep = 0;
  this->bActivated = false;
}


/**
 * this toggles the rotation direction
 */
void RepairStation::toggleRotation()
{
  this->animationStep *= -1;
}


/**
 * ticks the door
 * @param time: time since last tick
 */
void RepairStation::tick (float time)
{
  if( likely(this->getModel(0) != NULL))
    ((InteractiveModel*)this->getModel(0))->tick(time);

  if( !this->bActivated)
    return;

  if( this->isAnimationFinished())
  {
    this->animationCurrent = (this->animationCurrent + this->animationStep) % REPAIR_MAX_ANIMATIONS;
    this->setAnimation( this->animationCurrent, MD2_ANIM_ONCE);
  }
}




