

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific
   main-programmer: Patrick Boenzli
   co-programmer: Silvan Nellen
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "util/loading/factory.h"
#include "util/loading/load_param.h"

#include "interactive_model.h"


#include "gate.h"
#include "effects/explosion.h"






ObjectListDefinition(Gate);
CREATE_FACTORY(Gate);


#include "script_class.h"
CREATE_SCRIPTABLE_CLASS(Gate,
			addMethod("hide", Executor0<WorldEntity, lua_State*>(&WorldEntity::hide))
			    ->addMethod("unhide", Executor0<WorldEntity, lua_State*>(&WorldEntity::unhide))
			    ->addMethod("destroy", Executor0<Gate, lua_State*>(&Gate::destroy))
			    ->addMethod("setAbsCoor", Executor3<PNode, lua_State*,float,float,float>(&PNode::setAbsCoor))
			    ->addMethod("getAbsCoorX", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorX))
			    ->addMethod("getAbsCoorY", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorY))
			    ->addMethod("getAbsCoorZ", Executor0ret<PNode, lua_State*, float>(&PNode::getAbsCoorZ))
                       );



//! list of all different animations a std md2model supports
sAnim Gate::animationList[3] =
{
 // begin, end, fps, interruptable
  {   0,  7,  30,  0 },    //!< OPEN
  {   8, 15,  30,  0 },    //!< CLOSE
  {  16, 18,  30,  0 }     //!< DIE
};



Gate::Gate(const TiXmlElement* root)
{
  this->registerObject(this, Gate::_objectList);
  this->scale = 1.0f;
  this->actionRadius = 1.0;
  this->destroyed = false;

  if( root != NULL)
    this->loadParams(root);

  this->toList(OM_COMMON);
  this->bLocked = false;
  this->bOpen = false;

  this->loadMD2Texture("textures/wheel.jpg");
  this->loadModel("models/creatures/hypergate.md2", this->scale);

  this->setAnimation(GATE_CLOSE, MD2_ANIM_ONCE);
}


Gate::~Gate ()
{}



/**
 * loads the Settings of a MD2Creature from an XML-element.
 * @param root the XML-element to load the MD2Creature's properties from
 */
void Gate::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "action-radius", this, Gate, setActionRadius)
      .describe("sets the action radius of the door")
      .defaultValues(3.0);
  LoadParam(root, "scale", this, Gate, setScale)
      .describe("sets the scale of the door")
      .defaultValues(1.0);
}


/**
 * sets the animatin of this entity
 */
void  Gate::setAnimation(int animNum, int playbackMode)
{
  if( likely(this->getModel(0) != NULL))
    ((InteractiveModel*)this->getModel(0))->setAnimation(animationList[animNum].firstFrame,
                                                         animationList[animNum].lastFrame,
                                                         animationList[animNum].fps,
                                                         animationList[animNum].bStoppable,
                                                         playbackMode);
}


/**
 * ticks the door
 * @param time: time since last tick
 */
void Gate::tick (float time)
{
  if( likely(this->getModel(0) != NULL))
    ((InteractiveModel*)this->getModel(0))->tick(time);


  if( !this->bOpen)
  {
    if( this->checkOpen())
    {
      this->open();
    }
  }
  else
  {
    if( !this->checkOpen())
    {
      this->close();
    }
  }


}


/**
 * open the door
 */
void Gate::open()
{
  if( this->bLocked || this->destroyed)
    return;

  this->setAnimation(GATE_OPEN, MD2_ANIM_ONCE);
  this->bOpen = true;
}


/**
 * close the door
 */
void Gate::close()
{

  if( this->destroyed)
    return;

  this->setAnimation(GATE_CLOSE, MD2_ANIM_ONCE);
  this->bOpen = false;
}


void Gate::destroy()
{
  if( this->destroyed)
    return;

  this->setAnimation(GATE_DIE, MD2_ANIM_ONCE);

  Explosion::explode(this, Vector(this->getScaling()/160,this->getScaling()/160,this->getScaling()/160));


  this->destroyed = true;
}

#include "playable.h"
#include "generic_npc.h"

/**
 * checks if the door is open
 */
bool Gate::checkOpen()
{

  std::list<BaseObject*>::const_iterator it;
  WorldEntity* entity;
  float distance;

  // for all players
  for (ObjectList<Playable>::const_iterator it = Playable::objectList().begin();
       it != Playable::objectList().end();
       ++it)
  {
    entity = (*it);

    distance = fabs((this->getAbsCoor() - entity->getAbsCoor()).len());
    if( distance < this->actionRadius)
      return true;
  }


  for (ObjectList<GenericNPC>::const_iterator it = GenericNPC::objectList().begin();
       it != GenericNPC::objectList().end();
       ++it)
  {
    entity = (*it);

    distance = fabs((this->getAbsCoor() - entity->getAbsCoor()).len());
    if( distance < this->actionRadius)
      return true;
  }

  return false;
}



