/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: David Hasenfratz, Stefan Lienhard
   co-programmer:
*/

#include "movie_loader.h"

#include "movie_player.h"
#include "util/loading/factory.h"
#include "event_handler.h"
#include "graphics_engine.h"
#include "util/loading/load_param.h"
#include "state.h"




ObjectListDefinition(MovieLoader);
CREATE_FACTORY(MovieLoader);

MovieLoader::MovieLoader(const TiXmlElement* root)
{
  this->registerObject(this, MovieLoader::_objectList);

  movie_player = new MoviePlayer();
  this->loadParams(root);
}

MovieLoader::~MovieLoader()
{
  delete this->movie_player;
}



void MovieLoader::loadParams(const TiXmlElement* root)
{
  StoryEntity::loadParams(root);

  LoadParam(root, "movie", this, MovieLoader, loadMovie);
  LoadParam(root, "fps", this, MovieLoader, setFPS);
}

void MovieLoader::setFPS(float fps)
{
  this->movie_player->setFPS(fps);
}

void MovieLoader::loadMovie(const std::string& filename)
{
  movie_player->loadMovie(filename);
}


ErrorMessage MovieLoader::init() {  return ErrorMessage();}


ErrorMessage MovieLoader::loadData() {  return ErrorMessage();}


ErrorMessage MovieLoader::unloadData()
{
  this->unsubscribeEvents(ES_GAME);

  return ErrorMessage();
}

bool MovieLoader::start()
{
  EventHandler::getInstance()->pushState(ES_GAME);

  this->movie_player->start(0);

  this->bRunning = true;
  this->run();

  return true;
}

bool MovieLoader::stop()
{
  EventHandler::getInstance()->popState();

  this->bRunning = false;

  return true;
}

bool MovieLoader::pause() { return false; }
bool MovieLoader::resume() { return false; }

void MovieLoader::run()
{
  // first timestamp for t = 0
  this->lastFrame = SDL_GetTicks ();

  while( this->bRunning)
  {
    EventHandler::getInstance()->process();
    this->tick();
    this->draw();
  }
}

void MovieLoader::process(const Event &event) {}

void MovieLoader::draw() const
{
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);


  GraphicsEngine::enter2DMode();

  glPushAttrib(GL_ENABLE_BIT);
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, movie_player->getTexture());

  glColor3f(1.0, 1.0, 1.0);

  glBegin(GL_QUADS);
    glTexCoord2f(0.0f, 0.0f); glVertex2f( 0, 0);
    glTexCoord2f(0.0f, 1.0f); glVertex2f( 0, State::getResY());
    glTexCoord2f(1.0f, 1.0f); glVertex2f( State::getResX(), State::getResY());
    glTexCoord2f(1.0f, 0.0f); glVertex2f( State::getResX(), 0);
  glEnd();

  glPopAttrib();
  GraphicsEngine::leave2DMode();

  SDL_GL_SwapBuffers();
}


void MovieLoader::tick()
{
  // get timestamp
  currentFrame = SDL_GetTicks();

  // calculate time difference in milliseconds (Uint32)
  this->dt = currentFrame - this->lastFrame;
  // calculate time difference in seconds (float)
  this->dts = (float)this->dt / 1000.0f;

  movie_player->tick(dts);

  if (movie_player->getStatus() == STOP)
    this->bRunning = false;

  this->lastFrame = currentFrame;
}
